import random
import time
import numpy as np

# -----------------------
# Example 2.1:  Rolling Dice
# -----------------------

# Function to simulate a roll of a single die
def roll():
    return random.randint(1, 6)

# Function to roll d dice and find the probability P(total = k)
def probtotk(d, k, nreps):
    count = 0
    for rep in range(nreps):
        current_sum = 0
        for j in range(d):
            current_sum += roll()
        if current_sum == k:
            count += 1
    return count / nreps

# Call example
result = probtotk(3, 8, 1000)
print(result)

start_time = time.time()
probtotk(3, 8, 100000)
end_time = time.time()
t1 = end_time - start_time
print("Time taken:", t1)

# -----------------------
# Python equivalent of the First Improvement in R Code
# It turns out this is also faster (i.e. first improvement) in Python
# -----------------------

# Improved function to roll d dice and find P(total = k)
def probtotk(d, k, nreps):
    count = 0
    for rep in range(nreps):
        total = sum(random.choices(range(1, 7), k=d))
        if total == k:
            count += 1
    return count / nreps

# Call example
result = probtotk(3, 8, 1000)
print(result)

start_time = time.time()
probtotk(3, 8, 100000)
end_time = time.time()
t2 = end_time - start_time
print("Time taken:", t2)

# -----------------------
# Python equivalent of the Second Improvement in R Code
# It turns out this is also faster (i.e. second improvement) in Python
# -----------------------

# Function to simulate rolling nd dice
def roll(nd):
    return random.choices(range(1, 7), k=nd)

# Improved function with vectorized sum operations using numpy
def probtotk(d, k, nreps):
    sums = np.array([sum(roll(d)) for _ in range(nreps)])
    return np.mean(sums == k)

# Call example
result = probtotk(3, 8, 1000)
print(result)

start_time = time.time()
probtotk(3, 8, 100000)
end_time = time.time()
t3 = end_time - start_time
print("Time taken:", t3)

# -----------------------
# We don't provide a Pyhton equivalent of the Third Improvement in the R Code 
# -----------------------

# Function to simulate rolling nd dice
def roll(nd):
    return random.choices(range(1, 7), k=nd)

# Print time taken for the three methods
print(f"Method 1: {t1:.6f} seconds")
print(f"Method 2: {t2:.6f} seconds")
print(f"Method 3: {t3:.6f} seconds")

##################################
#Example 2.2: Dice Problem
##################################
# Function to simulate the dice problem
def dicesim(nreps):
    count1 = 0  # count of rolls with sum > 8
    count2 = 0  # count of rolls where first die < 3 and sum > 8
    
    # Simulate the dice roll nreps times
    for i in range(nreps):
        d = [random.randint(1, 6) for i in range(3)]  # Roll three dice
        if sum(d) > 8:
            count1 += 1
            if d[0] < 3:
                count2 += 1
    
    # Return the conditional probability
    return count2 / count1 if count1 != 0 else 0  # Added check to avoid division by zero

# Simulate and compute the probability with 1000 repetitions
result = dicesim(1000)
print(result)

##################################
#Section 2.3 Use of random for Simulating Events
##################################
# Simulate a coin toss
# random.random() generates a random number between 0 and 1
# If it's less than 0.5, we consider it as "heads"
heads = random.random() < 0.5

# Print the result of the coin toss
print(heads)

##################################
#Example: ALOHA Network
##################################

# Simulating the ALOHA Network

def ALOHAsim(p, q, nreps):
    countx2eq2 = 0
    countx1eq1 = 0
    countx1eq2 = 0
    countx2eq2givx1eq1 = 0
    
    for i in range(nreps):
        numsend = sum([1 for _ in range(2) if random.random() < p])
        
        if numsend == 1:
            X1 = 1
        else:
            X1 = 2
            
        if X1 == 2:
            countx1eq2 += 1
            
        numactive = X1 + (1 if X1 == 1 and random.random() < q else 0)
        
        if numactive == 1:
            X2 = 0 if random.random() < p else 1
        else:
            numsend = sum([1 for _ in range(2) if random.random() < p])
            X2 = 1 if numsend == 1 else 2
            
        if X2 == 2:
            countx2eq2 += 1
            
        if X1 == 1:
            countx1eq1 += 1
            if X2 == 2:
                countx2eq2givx1eq1 += 1
                
    print("P(X1 = 2):", countx1eq2/nreps)
    print("P(X2 = 2):", countx2eq2/nreps)
    print("P(X2 = 2 | X1 = 1):", countx2eq2givx1eq1/countx1eq1)

# Run the example
ALOHAsim(.4, .8, 1000)


##################################
#Example 2.4: Bus Ridership
##################################
# In this simulation, we consider a bus with multiple stops.
# At each stop, passengers can get off or get on.
# We want to determine the probability that the bus is empty after visiting the nth stop.
# In this example, n=100.

# Number of repetitions for the simulation
nreps = 10000
# Number of stops the bus visits
nstops = 100
# Initialize a counter for the number of times the bus is empty after nth stop
count = 0

# Start the simulation loop for nreps repetitions
for i in range(nreps):
    # Initialize passengers on the bus
    passengers = 0
    # Loop through each of the stops
    for j in range(nstops):
        # If there are passengers on the bus
        if passengers > 0:
            # For each passenger, check if they get off the bus with 20% probability
            for k in range(passengers):
                if random.random() < 0.2:
                    passengers -= 1
        # At each stop, new passengers can board the bus.
        # There's a 50% chance 0 passengers board, 40% chance 1 passenger boards, and 10% chance 2 passengers board.
        newpass = random.choices([0, 1, 2], weights=[0.5, 0.4, 0.1])[0]
        # Add the new passengers to the current passengers on the bus
        passengers += newpass
    # Check if the bus is empty after the nth stop and increase the counter if it is
    if passengers == 0:
        count += 1

# Print the probability that the bus is empty after the nth stop
print(count/nreps)

##################################
#Example 2.5: Board Game
##################################

# In this simulation, a player is simulating a board game.
# The board has 8 spaces numbered 1 through 8, and the player starts on a space by rolling a 6-sided die.
# If the player starts on space 3, they roll again and move that many spaces forward (the board is circular, 
# so moving from space 8 brings you to space 1).
# The goal is to estimate the probability that the player rolled the bonus roll 
#(from landing on space 3) given that they end up on space 4.

def boardsim(nreps):
    # Count of times the player ends up on space 4
    count4 = 0
    # Count of times the player gets the bonus roll and ends up on space 4
    countbonusgiven4 = 0
    
    # Start the simulation loop for nreps repetitions
    for _ in range(nreps):
        # The player rolls a die to determine their starting position
        position = random.randint(1, 6)
        # If they land on space 3, they get a bonus roll
        if position == 3:
            bonus = True
            # Roll the die again and move forward (using modulo to handle the circular board)
            position = (position + random.randint(1, 6)) % 8
        else:
            bonus = False # No bonus roll if not landing on space 3
        # Check if they ended up on space 4
        if position == 4:
            count4 += 1
            # If they got the bonus roll and ended up on space 4, increase the counter
            if bonus:
                countbonusgiven4 += 1
                
    # Return the probability that the player got the bonus roll given they ended up on space 4
    return countbonusgiven4 / count4

# Call the function to estimate the probability with 1000 repetitions
print(boardsim(1000))


##################################
#Example 2.6: Broken Rod
##################################

# In this simulation, a glass rod is dropped and breaks into random pieces.
# We want to estimate the probability that the smallest piece has a length below 0.02.

# Simulating the Broken Rod

# Function minpiece:
# It simulates a single event of the rod breaking into k pieces.
# It returns the length of the smallest piece.

def minpiece(k):
    # Generate random breakpoints for the rod. 
    # k-1 breakpoints will result in k pieces.
    breakpts = np.sort(np.random.rand(k-1))
    
    # Calculate the lengths of the pieces using the numpy diff function.
    # np.concatenate adds 0 and 1 to the breakpoints 
    # so we capture the lengths of all pieces including the first and last.
    lengths = np.diff(np.concatenate(([0], breakpts, [1])))
    
    # Return the length of the shortest piece
    return np.min(lengths)

def bkrod(nreps, k, q):
    # Use list comprehension to replicate the minpiece function nreps times
    # to get the lengths of the smallest pieces from each simulation.
    minpieces = [minpiece(k) for _ in range(nreps)]
    
    # Calculate and return the fraction of times the smallest piece 
    # was less than q in length.
    return np.mean(np.array(minpieces) < q)

# Call the function to estimate the probability with 1000 repetitions,
# breaking the rod into 5 pieces and checking for pieces shorter than 0.02.
print(bkrod(1000, 5, 0.02))


##################################
#Example: Toss a Coin Until k Consecutive Heads
##################################

# In this simulation, we are tossing a coin until we get k heads in a row.
# The objective is to estimate the probability that more than m tosses are needed
# to achieve k consecutive heads.

# Simulating Tossing a Coin

# Function ngtm:
# Simulates the process of tossing a coin repeatedly to get k consecutive heads.
# It estimates the probability that more than m tosses are needed for this.

def ngtm(k, m, nreps):
    count = 0
    # Run the simulation nreps times
    for _ in range(nreps):
        # Initialize counter for consecutive heads
        consech = 0
        # Toss the coin m times
        for _ in range(m):
            # Sample 0 (for tails) or 1 (for heads)
            toss = np.random.choice([0, 1])
            
            # If the toss results in a head, increase the counter for consecutive heads
            if toss:
                consech += 1
                
                # If we get k consecutive heads, stop this repetition
                if consech == k:
                    break
            else:
                # If the toss results in a tail, reset the counter for consecutive heads
                consech = 0
        
        # If, after m tosses, we didn't get k consecutive heads, increment the count
        if consech < k:
            count += 1
    
    # Return the fraction of simulations where more than m tosses were needed
    # to get k consecutive heads.
    return count/nreps

# Call the function with k=8 consecutive heads, m=10 maximum number of tosses,
# and 1000 repetitions of the simulation.
print(ngtm(k=8, m=10, nreps=1000))


