import matplotlib.pyplot as plt
from scipy.stats import binom, norm, poisson
import numpy as np
from math import comb  # to compute combinations

# Setting parameters for the binomial distribution
n, p = 5, 1/5
s = range(7)

# Plotting the binomial pmf
plt.stem(s, binom.pmf(s, n, p), markerfmt='o', basefmt=" ", linefmt="b-", label="Binomial")
# Adding Poisson probabilities as points
plt.scatter(s, poisson.pmf(s, n*p), color='red', label="Poisson")
plt.title("Poisson(1) vs Binomial(n=5, p=1/5) pmfs")
plt.ylabel("prob density")
plt.legend()
plt.show()

plt.clf() # Clear the previous plot

# Updating parameters for the binomial distribution
n, p = 10, 1/10
s_shifted = [i+0.04 for i in s]

# Plotting the binomial pmf
plt.stem(s, binom.pmf(s, n, p), markerfmt='o', basefmt=" ", 
linefmt="b-", label="Binomial")
# Adding Poisson probabilities as red shifted lines
plt.stem(s_shifted, poisson.pmf(s, n*p), markerfmt='o', basefmt=" ", 
linefmt="r-", label="Poisson")
plt.title("Binomial(n=10, p=1/10) vs Poisson(1) pmfs")
plt.ylabel("prob density")
plt.legend(loc="upper right")
plt.show()

plt.clf()

###############################################################
# Setting up the scenario for P(X>=85) when X follows Binomial(n=100,p=0.8)
n, p = 100, 0.8
x = np.arange(85, 101)  # equivalent to seq(85, 100, by=1) in R

# Calculating the exact probability using the binomial formula
prob = binom.pmf(x, n, p)
exact = prob.sum()
print("Exact:", exact)

# Another method to get P(X>=85) using binomial cdf
method_2 = 1 - binom.cdf(84, n, p)
print("Using binom.cdf:", method_2)

# Using the Normal approximation to approximate the binomial distribution
mu = n * p
sd = (n * p * (1-p))**0.5

# Without continuity correction
z = (85 - mu) / sd
approx1 = 1 - norm.cdf(z)
print("Approximation without continuity correction:", approx1)
print("Difference without continuity correction:", abs(exact - approx1))

# With continuity correction
z = (84.5 - mu) / sd
approx2 = 1 - norm.cdf(z)
print("Approximation with continuity correction:", approx2)
print("Difference with continuity correction:", abs(exact - approx2))

# Attempting the Poisson approximation to the binomial
lambda_val = n * p
approx3 = poisson.sf(84, lambda_val)  # sf is survival function, equivalent to 1-cdf
print("Poisson approximation:", approx3)
print("Difference with Poisson approximation:", abs(exact - approx3))
#Why is the Poisson approximation bad here?

###############################################################
# Setting up the scenario for P(X>=60) when X follows Binomial(n=100,p=0.55)
n, p = 100, 0.55
x = np.arange(60, 101)  # equivalent to seq(60, 100, by=1) in R

# Calculating the exact probability using the binomial formula
prob = [comb(100, xi) * (0.55**xi) * (0.45**(100-xi)) for xi in x]
exact = sum(prob)
print("Exact:", exact)

# Another method to get P(X>=60) using binomial cdf
method_2 = 1 - binom.cdf(60, n, p)
print("Using binom.cdf:", method_2)

# Using the Normal approximation to approximate the binomial distribution
mu = n * p
sd = (n * p * (1-p))**0.5

# Without continuity correction
z = (60 - mu) / sd
approx1 = 1 - norm.cdf(z)
print("Approximation without continuity correction:", approx1)
print("Difference without continuity correction:", abs(exact - approx1))

# With continuity correction
z = (59.5 - mu) / sd
approx2 = 1 - norm.cdf(z)
print("Approximation with continuity correction:", approx2)
print("Difference with continuity correction:", abs(exact - approx2))

