## Lecture Slides 6 R Examples

# Load necessary libraries
library(bayesrules)
library(tidyverse)

#Set Working Directory to Source File Location
library("rstudioapi")  # Load rstudioapi package
setwd(dirname(getActiveDocumentContext()$path)) # Set working directory to source file location
#getwd()  

# Custom print function for intervals (confidence and credible intervals), rounded to r decimals
print_int <- function(interval,r=3) {
  interval <- round(interval,r)
  return(paste0("(", interval[1], ",", interval[2], ")"))
}

#Plot of the Prior on the Odds of Success
# Define the function p(tau)
p_tau <- function(tau) {
  return(1 / (1 + tau)^2)
}

# Generate a sequence of tau values from 0 to 1
tau_vals <- seq(0, 10, length.out = 100)

# Compute p(tau) for each tau
p_vals <- p_tau(tau_vals)

# Plot the function
plot(tau_vals, p_vals, type = "l", col = "blue", lwd = 2,
     xlab = expression(tau), ylab = expression(p(tau)),
     main = expression(p(tau) == frac(1, (1 + tau)^2)))
# Add grid lines
grid()

##########################################################
#Plot of Beta(1/2, 1/2) distribution

# Define the Beta(1/2, 1/2) density function
beta_dens <- function(x) {
  dbeta(x, 1/2, 1/2)
}

# Generate a sequence of x values from 0 to 1
x_vals <- seq(0, 1, length.out = 100)

# Compute the corresponding Beta(1/2, 1/2) density values
y_vals <- beta_dens(x_vals)

# Plot the Beta(1/2, 1/2) distribution
plot(x_vals, y_vals, type = "l", col = "blue", lwd = 2,
     xlab = expression(theta), ylab = expression(p(theta)),
     main = "Beta(1/2, 1/2) distribution")
# Add grid lines for better readability
grid()

##########################################################
### Example 1 (Cabinet Duration Example in Slide 6)
##########################################################
# Data for the Cabinet Duration example:
# N is the number of cabinets for eleven Western European countries from 1945 to
# 1980 for annualized periods and dur is the average duration
N <- c(38,28,27,20,17,15,15,15,15,14,12)
# The sum of the vector N gives the total number of observations
dur <- c(0.833,1.070,1.234,1.671,2.065,2.080,2.114,2.168,2.274, 2.629,2.637)
# The sum of the vector N*dur gives the sum of all the durations 
# (of all the x_i's)

# A 90% quantile-based credible interval for theta (reciprocal of mean duration):
qgamma(c(0.05,.95),shape=sum(N), rate=sum(N*dur))

#With the flat (uninformative) prior
qgamma(c(0.05,.95),shape=sum(N)+1, rate=sum(N*dur))

##########################################################
# Modernized and Enhanced Code:
##########################################################

##########################################################
# Plot of the Prior on the Odds of Success
##########################################################

p_tau <- function(tau) {
  1 / (1 + tau)^2
}

# Generate and plot the function using ggplot2
tibble(tau = seq(0, 10, length.out = 100)) %>%
  mutate(p_tau = p_tau(tau)) %>%
  ggplot(aes(x = tau, y = p_tau)) +
  geom_line(color = "blue", linewidth = 1) +
  labs(x = expression(tau), y = expression(p(tau)),
       title = expression(p(tau) == frac(1, (1 + tau)^2))) +
  theme_minimal() +
  theme(panel.grid = element_line(color = "gray90"))

##########################################################
# Plot of Beta(1/2, 1/2) distribution
##########################################################

# Plot the Beta(1/2, 1/2) distribution using ggplot2
tibble(x = seq(0, 1, length.out = 100)) %>%
  mutate(y = dbeta(x, 1/2, 1/2)) %>%
  ggplot(aes(x = x, y = y)) +
  geom_line(color = "blue", linewidth = 1) +
  labs(x = expression(theta), y = expression(p(theta)),
       title = "Beta(1/2, 1/2) Distribution") +
  theme_minimal() +
  theme(panel.grid = element_line(color = "gray90"))

##########################################################
### Example 1 (Cabinet Duration Example in Slide 6)
##########################################################

# Data for the Cabinet Duration example
N <- c(38, 28, 27, 20, 17, 15, 15, 15, 15, 14, 12)
dur <- c(0.833, 1.070, 1.234, 1.671, 2.065, 2.080, 2.114, 2.168, 2.274, 2.629, 2.637)

# Calculate sum of N and sum of N*dur
tot_N <- sum(N)
tot_N_dur <- sum(N * dur)

# A 90% quantile-based credible interval for theta (reciprocal of mean duration)
CI_theta <- qgamma(c(0.05, 0.95), shape = tot_N, rate = tot_N_dur)
cat("90% CI for theta with uninformative prior:", print_int(CI_theta, 3), "\n",sep="")

# With the flat (uninformative) prior
CI_theta_flat <- qgamma(c(0.05, 0.95), shape = tot_N + 1, rate = tot_N_dur)
cat("90% CI for theta with flat prior:", print_int(CI_theta_flat, 3), "\n",sep="")


