## Lecture Slides 7 R Examples 
## Enhanced for Efficiency

# Load necessary libraries
library(bayesrules)  # For Bayesian examples and functions
library(tidyverse)   # For modern data manipulation
library(BaM)         # BaM book examples
library(bayesm)      # For Bayesian econometrics and multivariate normal functions
library(TeachingDemos)
library(pscl)

#Set Working Directory to Source File Location
library("rstudioapi")  # Load rstudioapi package
setwd(dirname(getActiveDocumentContext()$path)) # Set working directory to source file location
#getwd()  

####################################################
# Vague Priors with Normal Data
####################################################

# Midge data in mm:
x <- c(1.64, 1.70, 1.72, 1.74, 1.82, 1.82, 1.82, 1.90, 2.08)
n <- length(x)
xbar <- mean(x)
sd <- sd(x)

# HPD Interval for t-distribution
hpd95_t <- hpd(qt, df = n - 1, conf = 0.95)

# HPD Interval for mu (back-transformation)
hpd95_mu_vag <- hpd95_t * (sd / sqrt(n)) + xbar
hpd95_mu_vag %>% round(3)

# HPD Interval for sigma^2
hpd95_sig2_vag <- hpd(qigamma, alpha = 0.5 * (n - 1), beta = 0.5 * (n - 1) * (sd^2))
hpd95_sig2_vag %>% round(3)

####################################################
# Bivariate Normal Simulation Example
####################################################

# Function to generate posterior quantities for bivariate normal
BVN_post <- function(data_mat, alpha, beta, m, n0 = 5) {
  n <- nrow(data_mat)
  xbar <- colMeans(data_mat)
  S2 <- (n - 1) * var(data_mat)
  
  Wp.inv <- solve(beta) + S2 + ((n0 * n) / (n0 + n)) * (xbar - m) %*% t(xbar - m)
  Sigma <- solve(rWishart(1, alpha + n, solve(Wp.inv))[,,1])
  mu <- MASS::mvrnorm(1, (n0 * m + n * xbar) / (n0 + n), Sigma / (n0 + n))
  
  c(mu1 = mu[1], mu2 = mu[2], sig1 = Sigma[1, 1], sig2 = Sigma[2, 2], rho = Sigma[2, 1])
}

# Posterior summary function
norm_post_summary <- function(reps) {
  reps[,5] <- reps[,5]/sqrt(reps[,3]*reps[,4])
  reps <- apply(reps,2,sort)
  out.mat <- cbind("mean"=apply(reps,2,mean),
                   "std.err"=apply(reps,2,sd),
                   "95% HPD LL"=reps[25,],
                   "95% HPD UL"=reps[975,])
  return(out.mat)
}

# Simulate bivariate normal data
data_n10 <- MASS::mvrnorm(10, c(1, 3), matrix(c(1.0, 0.7, 0.7, 3.0), 2, 2))

# Generate posterior samples (matrix form)
rep_mat <- replicate(1000, BVN_post(data_n10, 3, matrix(c(10, 5, 5, 10), 2, 2), c(2, 2)), simplify = TRUE)

# Transpose the matrix and convert it to a tibble, then assign proper column names
rep_mat <- as_tibble(t(rep_mat))  # Convert to tibble after transposing

# Assign meaningful column names
colnames(rep_mat) <- c("mu1", "mu2", "sigma1", "sigma2", "rho")

# Summarize posterior
norm_post_summary(rep_mat)

####################################################
# Multivariate Normal Example with Health Data
####################################################

# Load NC county health data
data(nc.sub.dat)

# Specify priors and parameters
Alpha <- 3 + nrow(nc.sub.dat)  # Degrees of freedom
Beta.inv <- solve(diag(3) * 100)  # Inverse of Beta
m <- c(250, 16, 88)  # Prior mean vector
n0 <- 0.01  # Prior sample size

# Compute sample mean and variance-covariance matrix
xbar <- colMeans(nc.sub.dat)
S2 <- var(nc.sub.dat)

# Weighting factor for the mean
k <- (n0 * nrow(nc.sub.dat)) / (n0 + nrow(nc.sub.dat))

# Compute the posterior scale matrix
p.Beta <- solve(Beta.inv + S2 + k * (xbar - m) %*% t(xbar - m))

# Simulate Sigma matrices using the Wishart distribution
Sigma_sim <- replicate(10000, solve(rWishart(1, Alpha, Sigma = p.Beta)[,,1]), simplify = FALSE)
Sigma <- array(unlist(Sigma_sim), dim = c(3, 3, length(Sigma_sim)))

# Posterior mean and standard deviation of Sigma
Sigma_Mean <- apply(Sigma, c(1, 2), mean)
Sigma_SD <- apply(Sigma, c(1, 2), sd)

# Display mean and SD
Sigma_Mean
Sigma_SD

# Analytical mean of the inverse Wishart distribution
inv_wish_mean <- solve(p.Beta) / (Alpha - ncol(nc.sub.dat) - 1)
inv_wish_mean

# Simulate Mu values and calculate quantiles
Mu <- MASS::mvrnorm(5000, (n0 * m + nrow(nc.sub.dat) * xbar) / (n0 + nrow(nc.sub.dat)), Sigma_Mean / (n0 + nrow(nc.sub.dat)))
apply(Mu, 2, quantile, probs = c(0.01, 0.25, 0.50, 0.75, 0.99))

######################################################
# Example: National IQ Scores
######################################################

data(iq)
iq <- as.numeric(iq)
n <- length(iq)

# T-distribution for IQ scores
t.iq <- (iq - mean(iq)) / (sd(iq) / sqrt(n))
r.t <- rt(100000, df = n - 1) * (sd(iq) / sqrt(n)) + mean(iq)
quantile(r.t, c(0.01, 0.10, 0.25, 0.50, 0.75, 0.90, 0.99))

# Simulate sigma^2 from inverse gamma
r.Sigma_sq <- 1 / rgamma(100000, shape = (n + 1) / 2, rate = var(iq) * (n - 1) / 2)
quantile(sqrt(r.Sigma_sq), c(0.01, 0.10, 0.25, 0.50, 0.75, 0.90, 0.99))

