#AuxFunctions.r
#Some auxiliary functions for the methods in Givens & Hoeting

# modified from Dr Ryan Martin's code 

## PART I: Univariate Methods

####################################################
# Function to implement the BISECTION Method
####################################################
bisection <- function(f, a, b, eps = 1e-08, maxiter = 1000, verbose = FALSE, ...) {
  fa <- f(a, ...)
  fb <- f(b, ...)
  
  if(fa * fb > 0) {
    stop("Function values at the interval endpoints must be of opposite sign.")
  }
  
  x <- (a + b) / 2
  fx <- f(x, ...)
  t <- 0
  
  if(verbose) cat("\n")
  
  while(abs(fx) > eps && t < maxiter) {
    t <- t + 1
    if(fa * fx <= 0) {
      b <- x
      fb <- fx
    } else {
      a <- x
      fa <- fx
    }
    x.new <- (a + b) / 2
    fx <- f(x.new, ...)
    
    if(verbose) cat("[", t, "]", as.double(x.new), fx, "\n")
    
    if(abs(x.new - x) < eps) break
    x <- x.new
  }
  
  if(t >= maxiter) warning("Maximum number of iterations reached!")
  
  if(verbose) cat("\n")
  out <- list(solution = x, value = fx, iter = t)
  return(out)
}

####################################################
# Function to implement NEWTON'S Method
####################################################
newton <- function(f, df, x0, eps = 1e-08, maxiter = 1000, verbose = FALSE, ...) {
  x <- x0
  fx <- f(x, ...)
  dfx <- df(x, ...)
  t <- 0
  
  if(verbose) cat("\n")
  
  while(abs(fx) > eps && t < maxiter) {
    t <- t + 1
    if(dfx == 0) {
      warning("Derivative is zero. No solution found.")
      break
    }
    
    x.new <- x - fx / dfx
    fx.new <- f(x.new, ...)
    
    if(verbose) cat("[", t, "]", x.new, fx.new, "\n")
    
    if(abs(x.new - x) < eps) break
    x <- x.new
    fx <- fx.new
    dfx <- df(x, ...)
  }
  
  if(t >= maxiter) warning("Maximum number of iterations reached!")
  
  if(verbose) cat("\n")
  out <- list(solution = x, value = fx, iter = t)
  return(out)
}

####################################################
# Function to implement the SECANT Method
####################################################
secant <- function(f, x0, x1, eps = 1e-08, maxiter = 1000, verbose = FALSE, ...) {
  x.old <- x0
  x <- x1
  f.old <- f(x.old, ...)
  f.x <- f(x, ...)
  t <- 0
  
  if(verbose) cat("\n")
  
  while(abs(f.x) > eps && t < maxiter) {
    t <- t + 1
    
    if(x == x.old) {
      warning("Division by zero error in Secant method.")
      break
    }
    
    diff.ratio <- (f.x - f.old) / (x - x.old)
    
    if(diff.ratio == 0) {
      warning("Zero slope encountered in Secant method.")
      break
    }
    
    x.new <- x - f.x / diff.ratio
    f.new <- f(x.new, ...)
    
    if(verbose) cat("[", t, "]", x.new, f.new, "\n")
    
    if(abs(x.new - x) < eps) break
    
    x.old <- x
    f.old <- f.x
    x <- x.new
    f.x <- f.new
  }
  
  if(t >= maxiter) warning("Maximum number of iterations reached!")
  
  if(verbose) cat("\n")
  out <- list(solution = x, value = f.x, iter = t)
  return(out)
}

####################################################
# Function to implement FIXED POINT ITERATION Method
####################################################
fixed_pnt_iter <- function(F, x0, eps = 1e-08, maxiter = 1000, verbose = FALSE, ...) {
  x <- x0
  Fx <- F(x, ...)
  t <- 0
  
  if(verbose) cat("\n")
  
  while(abs(Fx - x) > eps && t < maxiter) {
    t <- t + 1
    x <- Fx
    Fx <- F(x, ...)
    
    if(verbose) cat("[", t, "]", x, Fx, "\n")
    
    if(abs(Fx - x) < eps) break
  }
  
  if(t >= maxiter) 
    {warning("Maximum number of iterations reached!")
  x=NA
  Fx=NA}
  
  if(verbose) cat("\n")
  out <- list(solution = x, value = Fx, iter = t)
  return(out)
}

##########################################################################
## PART 2: Multivariate Versions
##########################################################################
mv_newton <- function(f, df, x0, eps = 1e-08, maxiter = 1000, ...) {
  require(MASS) # Ensure the MASS package is loaded for ginv()
  
  x <- x0
  t <- 0
  fx <- f(x, ...)
  dfx <- df(x, ...)
  
  while(mean(abs(fx)) > eps && t < maxiter) {
    t <- t + 1
    # Use solve() if possible for efficiency; fall back to ginv() if solve() fails
    tryCatch({
      dx <- solve(dfx, fx)
    }, error = function(e) {
      dx <- ginv(dfx) %*% fx
    })
    
    x.new <- x - dx
    
    # Break if the change in x is below the threshold, indicating convergence
    if(mean(abs(x.new - x)) < eps) break
    
    x <- x.new
    fx <- f(x, ...)
    dfx <- df(x, ...)
  }
  
  if(t >= maxiter) warning("Maximum number of iterations reached!")
  
  out <- list(solution = x, value = fx, iter = t)
  return(out)
}

####################################################
# Steepest ascent for the Gamma MLE
####################################################
## Just for illustration -- not for HW, "real life", etc...
steepest_ascent <- function(dg, x0, alpha, eps = 1e-04, maxiter = 1000) {
  x <- x0
    path <- matrix(NA, nrow = maxiter + 1, ncol = length(x0))
    path[1, ] <- x
    
    for (t in 1:maxiter) {
      x.new <- x + alpha * dg(x)
      path[t + 1, ] <- x.new
      
      # Check for convergence with all elements satisfying the condition
      if (all(abs(x.new - x) / abs(x) < eps)) {
        path <- path[1:(t + 1), ]
        break
      }
      x <- x.new
    }
    
    return(list(x = x, path = path))
  }
  

####################################################
# A quasi-Newton method -- the BFGS rank-two update, 
# i.e. BFGS (Broyden–Fletcher–Goldfarb–Shanno) quasi-Newton method 
####################################################
bfgs <- function(f, df, x, M, alpha=1, eps=1e-08, maxiter=1000, hess=FALSE, ...) {
  if(any(eigen(M)$values >= 0)) stop("Starting quasi-hessian should be negative definite!")
  
  t <- 0
  fx <- f(x, ...)
  dfx <- df(x, ...)
  MM <- solve(M)
  
  repeat {
    t <- t + 1
    beta <- alpha
    ascent <- FALSE
    while(!ascent) {
      z <- -beta * MM %*% dfx
      x.new <- x + z
      fx.new <- f(x.new, ...)
      if(fx.new < fx) {
        beta <- beta / 2
      } else {
        ascent <- TRUE
      }
    }
    
    dfx.new <- df(x.new, ...)
    if(mean(abs(dfx.new)) < eps || t >= maxiter) {
      if(t >= maxiter) warning("maximum number of iterations reached!")
      break
    }
    
    y <- dfx.new - dfx
    s <- x.new - x
    rho <- 1 / as.numeric(t(y) %*% s)
    
    # Ensure s and y are treated as column vectors for outer product calculations
    s <- matrix(s, length(s), 1)
    y <- matrix(y, length(y), 1)
    
    V <- diag(length(x)) - rho * (s %*% t(y))
    MM <- V %*% MM %*% t(V) + rho * (s %*% t(s))
    
    x <- x.new
    fx <- fx.new
    dfx <- dfx.new
  }
  
  # Hessian calculation at the final estimate if requested
  if(hess) {
    neg.ddfx <- "Placeholder for Hessian computation. Implement if required."
  } else {
    neg.ddfx <- NULL
  }
  
  return(list(iter=t, x=x, fx=fx, dfx=dfx, neg.ddfx=neg.ddfx))
}

####################################################
# Gauss-Newton Method 
####################################################

gauss_newton <- function(y, f, df, x, eps=1e-08, maxiter=1000, ...) {
  if(!exists("ginv")) library(MASS)
  fx <- f(x, ...)
  dfx <- df(x, ...)
  t <- 0
  repeat {
    t <- t + 1
    x.new <- x + ginv(t(dfx) %*% dfx) %*% t(dfx) %*% (y - fx)
    if(mean(abs(x.new - x)) < eps || t >= maxiter) {
      if(t >= maxiter) warning("Maximum number of iterations reached!")
      break
    }
    x <- x.new
    fx <- f(x.new, ...)
    dfx <- df(x.new, ...)
  }
  return(list(x=as.numeric(x.new), fx=f(x.new, ...), iter=t))
}

####################################################
# Generic code for numerical integration (quadrature) ##
# Only Riemann, Trapezoid, or Simpson is implemented
####################################################
#  x = grid of (equispaced) points over which to integrate
# fx = value of the function f(x) at those grid points
#  m = degree of the approximating polynomial (only m = 0, 1, 2 is allowed)
myint <- function(fx, x, m=2) {
  h <- diff(x)[1]
  if(!(m %in% c(0, 1, 2))) stop("Only Riemann, Trapezoid, or Simpson is implemented!")
  
  switch(m + 1,
         { # Riemann sum (m=0)
           intf <- h * sum(fx[-length(fx)])
         },
         { # Trapezoidal rule (m=1)
           intf <- h * sum(fx[-1] + fx[-length(fx)]) / 2
         },
         { # Simpson's rule (m=2)
           if(length(fx) %% 2 == 1) stop("Simpson's rule requires an even number of intervals!")
           oddSum <- sum(fx[seq(3, length(fx)-2, by=2)])
           evenSum <- sum(fx[seq(2, length(fx)-1, by=2)])
           intf <- h * (fx[1] + fx[length(fx)] + 2 * oddSum + 4 * evenSum) / 3
         }
  )
  return(intf)
}

############################################################################
## --- General Code for the Metropolis-Hastings Algorithm --- ##
############################################################################

#    x0 = starting point of the chain (possibly a vector)
#     f = target density function
# dprop = density function for the proposal distribution
# rprop = proposal distribution
#     N = Monte Carlo sample size
#     B = burn-in length

mh <- function(x0, f, dprop, rprop, N, B = 0) {
  if (B > N) {
    stop("Burn-in length B cannot be greater than the sample size N.")
  }
  
  # Ensure x0 is a numeric vector
  if (!is.numeric(x0)) {
    stop("Starting point x0 must be a numeric vector.")
  }
  
  # Initialize the chain
  x <- matrix(NA, nrow = N + B, ncol = length(x0))
  fx <- numeric(N + B) # Preallocate for efficiency
  x[1, ] <- x0
  fx[1] <- f(x0)
  ct <- 0 # Acceptance count
  
  # Metropolis-Hastings sampler
  for (i in 2:(N + B)) {
    u <- rprop(x[i - 1, ])
    fu <- f(u)
    r <- log(fu) + log(dprop(x[i - 1, ], u)) - log(fx[i - 1]) - log(dprop(u, x[i - 1, ]))
    R <- min(exp(r), 1)
    
    if (runif(1) <= R) {
      ct <- ct + 1
      x[i, ] <- u
      fx[i] <- fu
    } else {
      x[i, ] <- x[i - 1, ]
      fx[i] <- fx[i - 1]
    }
  }
  
  # Prepare the output
  out <- list(
    x = x[-(1:B), ],      # Chain without burn-in
    fx = fx[-(1:B)],      # Target density evaluations without burn-in
    acc_rate = ct / (N + B)  # Acceptance rate
  )
  
  return(out)
}