#########################################################################
# COMPUTATIONAL STATISTICS
# by Geof Givens and Jennifer Hoeting
# CHAPTER 4 Examples
#########################################################################

#Set Working Directory to Source File Location
library("rstudioapi")                               # Load rstudioapi package
setwd(dirname(getActiveDocumentContext()$path))     # Set working directory to source file location
#getwd()   

#some functions for the methods used in G&H
source("AuxFunctions.r")

### Trivial Example (in Notes)
theta<-1
for (i in 1:20)
{
  print(c(i,theta))
  theta<-2*theta/(5*theta+1)
}

#########################################################################
### Example 4.2 EM Algorithm (Peppered Moths)
#########################################################################

# x = observed phenotype counts (carbonaria, insularia, typica)
# n = expected genotype frequencies (CC, CI, CT, II, IT, TT)
# p = allele probabilities (carbonaria, insularia, typica)
# iter = number of iterations
# alle_e = computes expected genotype frequencies
# alle_m = computes allele probabilities
#########################################################################

# Define the observed phenotype counts for carbonaria, insularia, and typica
x <- c(carbonaria = 85, insularia = 196, typica = 341)
# Initialize expected genotype frequencies and allele probabilities
n <- rep(0, 6)
p <- rep(1/3, 3)  # Equal probabilities for carbonaria, insularia, and typica
iter <- 40  # Number of iterations

# Function to compute expected genotype frequencies based on allele probabilities
alle_e <- function(x, p) {
    n_cc <- (x['carbonaria'] * (p[1]^2)) / (p[1]^2 + 2*p[1]*p[2] + 2*p[1]*p[3])
    n_ci <- (2 * x['carbonaria'] * p[1] * p[2]) / (p[1]^2 + 2*p[1]*p[2] + 2*p[1]*p[3])
    n_ct <- (2 * x['carbonaria'] * p[1] * p[3]) / (p[1]^2 + 2*p[1]*p[2] + 2*p[1]*p[3])
    n_ii <- (x['insularia'] * (p[2]^2)) / (p[2]^2 + 2*p[2]*p[3])
    n_it <- (2 * x['insularia'] * p[2] * p[3]) / (p[2]^2 + 2*p[2]*p[3])
    n_tt <- x['typica']
    n <- c(n_cc, n_ci, n_ct, n_ii, n_it, n_tt)
    return(n)
}

# Function to compute allele probabilities based on expected genotype frequencies
alle_m <- function(x, n) {
    p_c <- (2 * n[1] + n[2] + n[3]) / (2 * sum(x))
    p_i <- (2 * n[4] + n[5] + n[2]) / (2 * sum(x))
    p_t <- (2 * n[6] + n[3] + n[5]) / (2 * sum(x))
    p <- c(p_c, p_i, p_t)
    return(p)
}

# Main EM algorithm loop
for (i in 1:iter) {
    n <- alle_e(x, p)
    p <- alle_m(x, n)
}

# Output the final estimate for allele probabilities
p  # Final allele probabilities for carbonaria, insularia, and typica

#########################################################################
### Example 4.6 -- SEM Algorithm (Peppered Moths)
#########################################################################

#########################################################################
# x = observed phenotype counts (carbonaria, insularia, typica)
# n = expected genotype frequencies (CC, CI, CT, II, IT, TT)
# p = allele probabilities (carbonaria, insularia, typica)
# p_EM = em algorithm estimates
# n_EM = em algorithm estimates
# iter = number of iterations
# alle_e = computes expected genotype frequencies
# alle_m = computes allele probabilities
#########################################################################

# Initial values
geno_cts = c(85, 196, 341) # Counts of each genotype
alle_cts = rep(0, 6) # Placeholder for expected allele counts
iter = 40 # Number of iter for the EM algorithm
alle_freqs = c(0.07, 0.19, 0.74) # Initial allele frequencies
p_EM = alle_freqs # Placeholder for updated frequencies in EM
theta_mat = matrix(0, 3, 3) # Placeholder for some calculation matrix
psi_vec = rep(0, 3) # Placeholder for another calculation vector
r_mat = matrix(0, 3, 3) # Placeholder for yet another calculation matrix

# Calculate expected allele frequencies based on current estimates
alle_e <- function(x, p) {
    # Break down the formula into smaller, understandable parts
    denominator.ccit = (p[1]^2) + 2*p[1]*p[2] + 2*p[1]*p[3]
    denominator.iit = (p[2]^2) + 2*p[2]*p[3]
    
    # Calculate expected counts for each genotype
    n_cc = (x[1] * (p[1]^2)) / denominator.ccit
    n_ci = (2 * x[1] * p[1] * p[2]) / denominator.ccit
    n_ct = (2 * x[1] * p[1] * p[3]) / denominator.ccit
    n_ii = (x[2] * (p[2]^2)) / denominator.iit
    n_it = (2 * x[2] * p[2] * p[3]) / denominator.iit
    
    # Combine calculated counts into a single vector, with x[3] representing n_tt
    n = c(n_cc, n_ci, n_ct, n_ii, n_it, x[3])
    
    return(n)
}


# Update allele frequency estimates based on observed data
alle_m <- function(x, n) {
    # Calculate allele frequencies directly, improving readability and efficiency
    total_alleles = 2 * sum(x) # Total alleles in the population
    p_c = (2 * n[1] + n[2] + n[3]) / total_alleles
    p_i = (2 * n[4] + n[5] + n[2]) / total_alleles
    p_t = (2 * n[6] + n[3] + n[5]) / total_alleles
    
    # Combine new allele frequencies into a single vector
    p = c(p_c, p_i, p_t)
    
    return(p)
}


# Initialize convergence criteria
eps <- 1e-12 # Threshold for determining convergence
prev_p_EM <- p_EM # Store initial allele frequencies for comparison

# Loop through EM algorithm for a maximum of 'iter' iter
for (i in 1:iter) {
    # E-step: Calculate expected counts based on current estimates
    n_EM = alle_e(geno_cts, p_EM)
    
    # M-step: Update allele frequency estimates based on expected counts
    p_EM = alle_m(geno_cts, n_EM)
    
    # Check for convergence: If the change in allele frequencies is small enough, stop the loop
    if (max(abs(p_EM - prev_p_EM)) < eps) {
        cat(sprintf("Convergence achieved after %d iter.\n", i))
        break
    }
    
    # Update previous allele frequencies for the next iteration's comparison
    prev_p_EM = p_EM
}

# If loop completes without convergence notification
if (i == iter) {
    cat(sprintf("Reached maximum iter (%d) without convergence.\n", iter))
}

n_EM
p_EM

# Initial setup for theta matrix based on allele frequencies
init_Theta <- function(p, p_EM) {
    theta <- matrix(rep(p_EM, length(p) * length(p)), nrow = length(p))
    diag(theta) <- p
    return(theta)
}

theta <- init_Theta(alle_freqs, p_EM) # Use the function to initialize theta

# Main loop for the EM algorithm iter
for(t in 1:5){
    n <- alle_e(geno_cts, alle_freqs)
    p_hat <- alle_m(geno_cts, n)
    
    for(j in 1:length(alle_freqs)){
        # Update diagonal of theta with p_hat values for allele frequencies
        theta[j, j] <- p_hat[j]
        n <- alle_e(geno_cts, theta[, j])
        psi <- alle_m(geno_cts, n)
        
        # Calculate r using vectorized operations
        r_mat[, j] <- (psi - p_EM) / (theta[j, j] - p_EM[j])
    }
    
    p <- p_hat # Update p with the new estimates from p_hat
}

p

# Function to compute the inverse Fisher information matrix
comp_iy_hat <- function(n_EM, p_EM) {
    iy_hat <- matrix(0, 2, 2)
    
    # Diagonal elements based on allele frequency estimates and expected counts
    iy_hat[1, 1] <- ((2 * n_EM[1] + n_EM[2] + n_EM[3]) / (p_EM[1]^2) +
                         (2 * n_EM[6] + n_EM[3] + n_EM[5]) / (p_EM[3]^2))
    iy_hat[2, 2] <- ((2 * n_EM[4] + n_EM[5] + n_EM[2]) / (p_EM[2]^2) +
                         (2 * n_EM[6] + n_EM[3] + n_EM[5]) / (p_EM[3]^2))
    
    # Off-diagonal elements
    iy_hat[1, 2] <- iy_hat[2, 1] <- (2 * n_EM[6] + n_EM[3] + n_EM[5]) / (p_EM[3]^2)
    
    return(iy_hat)
}

# Function to compute standard errors and correlations
comp_se_cor <- function(iy_hat, r) {
    # Computes variance-covariance matrix
    var_hat <- solve(iy_hat) %*% (diag(2) + t(r[-3, -3]) %*% solve(diag(2) - t(r[-3, -3])))
    
    # Standard errors for each parameter
    sd_hat <- sqrt(diag(var_hat))
    sd_hat <- c(sd_hat, sqrt(sum(var_hat))) # Append sd for combined parameters
    
    # Correlation estimates
    cor_hat <- c(
        var_hat[1, 2] / (sd_hat[1] * sd_hat[2]),
        (-var_hat[1, 1] - var_hat[1, 2]) / (sd_hat[1] * sd_hat[3]),
        (-var_hat[2, 2] - var_hat[1, 2]) / (sd_hat[2] * sd_hat[3])
    )
    
    list(sd_hat = sd_hat, cor_hat = cor_hat)
}

# Example usage
iy_hat <- comp_iy_hat(n_EM, p_EM)
res <- comp_se_cor(iy_hat, r_mat)

# Output the res
res$sd_hat  # Standard error estimates for pc, pi, pt
res$cor_hat # Correlation estimates for pairs (pc,pi), (pc,pt), and (pi,pt)

#########################################################################
### Example 4.6 Bootstrap
#########################################################################

#########################################################################
# x = observed phenotype counts (carbonaria, insularia, typica)
# n = expected genotype frequencies (CC, CI, CT, II, IT, TT)
# p = allele probabilities (carbonaria, insularia, typica)
# iter = number of iterations
# theta = allele probabilities for psuedo-data
# alle_e = computes expected genotype frequencies
# alle_m = computes allele probabilities
#########################################################################

# Initial Values
geno_cts = c(85, 196, 341) # Counts of each genotype
alle_cts = rep(0, 6) # Placeholder for expected allele counts
p <- rep(1/3, 3)
iter <- 40
theta_mat <- matrix(0, 3, 10000)
set.seed(1)

# Expectation step function
# Calculates expected allele frequencies based on current probability estimates
alle_e <- function(x, p) {
    # Validate inputs
    if (length(x) != 3 || length(p) != 3) {
        stop("Input vectors 'x' and 'p' must both have a length of 3.")
    }
    
    # Vectorized computation for efficiency
    denominators <- c((p[1]^2) + 2*p[1]*p[2] + 2*p[1]*p[3], (p[2]^2) + 2*p[2]*p[3], 0)
    n_cc <- x[1] * (p[1]^2) / denominators[1]
    n_ci <- 2 * x[1] * p[1] * p[2] / denominators[1]
    n_ct <- 2 * x[1] * p[1] * p[3] / denominators[1]
    n_ii <- x[2] * (p[2]^2) / denominators[2]
    n_it <- 2 * x[2] * p[2] * p[3] / denominators[2]
    
    # Update n vector directly
    n <- c(n_cc, n_ci, n_ct, n_ii, n_it, x[3])
    return(n)
}

# Maximization step function
# Updates the probability estimates based on expected counts
alle_m <- function(x, n) {
    # Validate inputs
    if (length(x) != 3 || length(n) != 6) {
        stop("Input vector 'x' must have a length of 3 and 'n' a length of 6.")
    }
    
    # Use vectorized operations for efficient computation
    sums <- sum(x) * 2
    p <- c((2*n[1] + n[2] + n[3]) / sums, (2*n[4] + n[5] + n[2]) / sums, 
           (2*n[6] + n[3] + n[5]) / sums)
    return(p)
}

# Run EM algorithm once for initial estimates
for(i in 1:iter){
    n <- alle_e(geno_cts, p)
    p <- alle_m(geno_cts, n)
}

n
p

theta_mat[, 1] <- p

# Simulation loop for allele frequencies
total <- sum(geno_cts) # Total count to avoid repeated computation

for(j in 2:10000){
    n_c <- rbinom(1, total, geno_cts[1]/total)
    n_i <- rbinom(1, total - n_c, geno_cts[2]/(total - geno_cts[1]))
    n_t <- total - n_c - n_i
    x_new <- c(n_c, n_i, n_t)
    
    # Reset p for each simulation
    p <- rep(1/3, 3)
    
    # EM algorithm for new sample
    for(i in 1:iter){
        n <- alle_e(x_new, p)
        p <- alle_m(x_new, n)
    }
    theta_mat[, j] <- p
}

# Compute standard deviations and correlations efficiently
sd_hat <- apply(theta_mat, 1, sd)
cor_hat = c(cor(theta[1,],theta[2,]), cor(theta[1,],theta[3,]),
            cor(theta[2,],theta[3,]))

# Output
list(
    theta_first_col = theta_mat[, 1], # EM estimate for allele probabilities (p_c, p_i, p_t)
    sd_hat = sd_hat,  # Standard error estimates (p_c, p_i, p_t)
    cor_hat = cor_hat # Correlation estimates ({p_c,p_i}, {p_c,p_t}, {p_i,p_t})
)

#########################################################################
### Example 4.9 EM Gradient Algorithm (Peppered Moths)
#########################################################################

#########################################################################
# x = observed phenotype counts (carbonaria, insularia, typica)
# n = expected genotype frequencies (CC, CI, CT, II, IT, TT)
# p = allele probabilities (carbonaria, insularia, typica)
# iter = number of iterations
# prob_vals = allele probabilities (used for plotting)
# alpha_def = default scaling parameter
# alle_e = computes expected genotype frequencies
# alle_l = computes log likelihood
# Q_prime = computes the gradient of Q
# Q_2prime = computes the hessian of Q
#########################################################################

## Initial Values
x = c(85, 196, 341)
n = rep(0,6)
p = rep(1/3,3)
iter = 40
prob_vals = matrix(0,3,iter+1)
prob_vals[,1] = p
alpha_def = 2

## Functions
alle_e = function(x,p){
    n_cc = (x[1]*(p[1]^2))/((p[1]^2)+2*p[1]*p[2]+2*p[1]*p[3])
    n_ci = (2*x[1]*p[1]*p[2])/((p[1]^2)+2*p[1]*p[2]+2*p[1]*p[3])
    n_ct = (2*x[1]*p[1]*p[3])/((p[1]^2)+2*p[1]*p[2]+2*p[1]*p[3])
    n_ii = (x[2]*(p[2]^2))/((p[2]^2)+2*p[2]*p[3])
    n_it = (2*x[2]*p[2]*p[3])/((p[2]^2)+2*p[2]*p[3])
    n = c(n_cc,n_ci,n_ct,n_ii,n_it,x[3])
    return(n)
}

alle_l = function(x,p){
    l = ( x[1]*log(2*p[1] - p[1]^2) + x[2]*log(p[2]^2 + 2*p[2]*p[3]) +
          2*x[3]*log(p[3]) )
    return(l)
}

Q_prime = function(n,p){
    da = (2*n[1]+n[2]+n[3])/(p[1]) - (2*n[6]+n[3]+n[5])/(p[3])
    db = (2*n[4]+n[5]+n[2])/(p[2]) - (2*n[6]+n[3]+n[5])/(p[3])
    dQ = c(da,db)
    return(dQ)
}

Q_2prime = function(n,p){
    da2 = -(2*n[1]+n[2]+n[3])/(p[1]^2) - (2*n[6]+n[3]+n[5])/(p[3]^2)
    51
    db2 = -(2*n[4]+n[5]+n[2])/(p[2]^2) - (2*n[6]+n[3]+n[5])/(p[3]^2)
    dab = -(2*n[6]+n[3]+n[5])/(p[3]^2)
    d2Q = matrix(c(da2,dab,dab,db2), nrow=2, byrow=TRUE)
    return(d2Q)
}

## Main
l_old = alle_l(x,p)
for(i in 1:iter){
    alpha = alpha_def
    n = alle_e(x,p)
    p_new = p[1:2] - alpha*solve(Q_2prime(n,p))%*%Q_prime(n,p)
    p_new[3] = 1 - p_new[1] - p_new[2]
    if(all(p_new > 0) && all(p_new < 1)){l_new = alle_l(x,p_new)}
      # REDUCE ALPHA UNTIL A CORRECT STEP IS REACHED
        while(any(p_new < 0) || any(p_new > 1) || l_new < l_old){
        alpha = alpha/2
        p_new = p[1:2] - alpha*solve(Q_2prime(n,p))%*%Q_prime(n,p)
        p_new[3] = 1 - p_new[1] - p_new[2]
        if(all(p_new > 0) && all(p_new < 1)){l_new = alle_l(x,p_new)}
    }
    p = p_new
    prob_vals[,i+1] = p
    l_old = l_new
}

## Output
p # Final Estimate for Allele Probabilities (p_c, p_i, p_t)

## Plot of Convergence
pb = seq(0.001,0.4,length=100)
c = outer(pb,pb,function(a,b){1-a-b})
pbs = matrix(0,3,10000)
pbs[1,] = rep(pb,100)
pbs[2,] = rep(pb,each=100)
pbs[3,] = as.vector(c)
z = matrix(apply(pbs,2,alle_l,x=x),100,100)

contour(pb,pb,z,nlevels=20)
for(i in 1:iter){
    segments(prob_vals[1,i],prob_vals[2,i],prob_vals[1,i+1],
    prob_vals[2,i+1],lty=2)
}

#########################################################################
### Example 4.10 Aitken Acceleration (Peppered Moths)
#########################################################################

#########################################################################
# x = observed phenotype counts (carbonaria, insularia, typica)
# n = expected genotype frequencies (CC, CI, CT, II, IT, TT)
# p = allele probabilities (carbonaria, insularia, typica)
# iter = number of iterations
# prob_vals = allele probabilities (used for plotting)
# alpha_def = default scaling parameter
# alle_e = computes expected genotype frequencies
# alle_m = computes allele probabilities
# alle_l = computes log likelihood
# alle_iy = computes the complete information
# alle_l_2prime = computes the hessian of the log likelihood
#########################################################################

## Initial Values
x = c(85, 196, 341)
n = rep(0,6)
p = rep(1/3,3)
iter = 40
prob_vals = matrix(0,3,iter+1)
prob_vals[,1] = p
alpha_def = 2

## Functions
alle_e = function(x,p){
    n_cc = (x[1]*(p[1]^2))/((p[1]^2)+2*p[1]*p[2]+2*p[1]*p[3])
    n_ci = (2*x[1]*p[1]*p[2])/((p[1]^2)+2*p[1]*p[2]+2*p[1]*p[3])
    n_ct = (2*x[1]*p[1]*p[3])/((p[1]^2)+2*p[1]*p[2]+2*p[1]*p[3])
    n_ii = (x[2]*(p[2]^2))/((p[2]^2)+2*p[2]*p[3])
    n_it = (2*x[2]*p[2]*p[3])/((p[2]^2)+2*p[2]*p[3])
    n = c(n_cc,n_ci,n_ct,n_ii,n_it,x[3])
    return(n)
}

alle_m = function(x,n){
    p_c = (2*n[1]+n[2]+n[3])/(2*sum(x))
    p_i = (2*n[4]+n[5]+n[2])/(2*sum(x))
    p_t = (2*n[6]+n[3]+n[5])/(2*sum(x))
    p = c(p_c,p_i,p_t)
    return(p)
}

alle_l = function(x,p){
    l = ( x[1]*log(2*p[1] - p[1]^2) + x[2]*log(p[2]^2 + 2*p[2]*p[3]) +
      2*x[3]*log(p[3]) )
    return(l)
}

alle_iy = function(n,p){
    iy_hat=matrix(0,2,2)
    iy_hat[1,1] = ((2*n[1]+n[2]+n[3])/(p[1]^2) +
       (2*n[6]+n[3]+n[5])/(p[3]^2))
    iy_hat[2,2] = ((2*n[4]+n[5]+n[2])/(p[2]^2) +
       (2*n[6]+n[3]+n[5])/(p[3]^2))
    iy_hat[1,2] = iy_hat[2,1] = (2*n[6]+n[3]+n[5])/(p[3]^2)
    return(iy_hat)
}

alle_l_2prime = function(x,p){
    l_2prime = matrix(0,2,2)
    l_2prime[1,1] = ( (-x[1]*(2-2*p[1])^2)/((2*p[1]-p[1]^2)^2) -
      2*x[1]/(2*p[1]-p[1]^2) -
      (4*x[2])/((-2*p[1]-p[2]+2)^2) -
      2*x[3]/(p[3]^2))
    l_2prime[2,2] = ( (-4*x[2]*p[3]^2)/((p[2]^2 + 2*p[2]*p[3])^2) -
      2*x[2]/(p[2]^2 + 2*p[2]*p[3]) -
      2*x[3]/(p[3]^2))
    l_2prime[1,2] = ((-2*x[2])/((-2*p[1]-p[2]+2)^2) -
      2*x[3]/(p[3]^2))
    l_2prime[2,1] = l_2prime[1,2]
    return(l_2prime)
}

## Main
l_old = alle_l(x,p)
for(i in 1:iter){
    alpha = alpha_def
    n = alle_e(x,p)
    p_EM = alle_m(x,n)
    p_new = (p[1:2] - alpha*solve(alle_l_2prime(x,p))%*%
      alle_iy(n,p)%*%(p_EM[1:2]-p[1:2]))
    p_new[3] = 1 - p_new[1] - p_new[2]
    if(all(p_new > 0) && all(p_new < 1)){l_new = alle_l(x,p_new)}
    # REDUCE ALPHA UNTIL A CORRECT STEP IS REACHED
    while(any(p_new < 0) || any(p_new > 1) || l_new < l_old){
        alpha = alpha/2
        p_new = (p[1:2] - alpha*solve(alle_l_2prime(x,p))%*%
         alle_iy(n,p)%*%(p_EM[1:2]-p[1:2]))
        p_new[3] = 1 - p_new[1] - p_new[2]
        if(all(p_new > 0) && all(p_new < 1)){l_new = alle_l(x,p_new)}
    }
    p = p_new
    prob_vals[,i+1] = p
    l_old = l_new
}

## Output
p  # Final Estimate for Allele Probabilities (p_c, p_i, p_t)

## Plot of Convergence
pb = seq(0.001,0.4,length=100)
c = outer(pb,pb,function(a,b){1-a-b})
pbs = matrix(0,3,10000)
pbs[1,] = rep(pb,100)
pbs[2,] = rep(pb,each=100)
pbs[3,] = as.vector(c)
z = matrix(apply(pbs,2,alle_l,x=x),100,100)

contour(pb,pb,z,nlevels=20)
for(i in 1:iter){
    segments(prob_vals[1,i],prob_vals[2,i],prob_vals[1,i+1],
      prob_vals[2,i+1],lty=2)
}

#########################################################################
### Example 4.11 Quasi-Newton Acceleration (Peppered Moths)
#########################################################################

#########################################################################
# x = observed phenotype counts (carbonaria, insularia, typica)
# n = expected genotype frequencies (CC, CI, CT, II, IT, TT)
# p = allele probabilities (carbonaria, insularia, typica)
# iter = number of iterations
# m = approximation of the hessian of the log likelihood
# b = update information for m
# prob_vals = allele probabilities (used for plotting)
# alpha_def = default scaling parameter
# alle_e = computes expected genotype frequencies
# alle_l = computes log likelihood
# Q_prime = computes the gradient of Q
# Q_2prime = computes the hessian of Q
#########################################################################

## Initial Values
x = c(85, 196, 341)
n = rep(0,6)
p = rep(1/3,3)
iter = 20
m = matrix(0,2,2)
b = matrix(0,2,2)
prob_vals = matrix(0,3,iter+1)
prob_vals[,1] = p
alpha_def = 2

## Functions
alle_e = function(x,p){
    n_cc = (x[1]*(p[1]^2))/((p[1]^2)+2*p[1]*p[2]+2*p[1]*p[3])
    n_ci = (2*x[1]*p[1]*p[2])/((p[1]^2)+2*p[1]*p[2]+2*p[1]*p[3])
    n_ct = (2*x[1]*p[1]*p[3])/((p[1]^2)+2*p[1]*p[2]+2*p[1]*p[3])
    n_ii = (x[2]*(p[2]^2))/((p[2]^2)+2*p[2]*p[3])
    n_it = (2*x[2]*p[2]*p[3])/((p[2]^2)+2*p[2]*p[3])
    n = c(n_cc,n_ci,n_ct,n_ii,n_it,x[3])
    return(n)
}

alle_l = function(x,p){
    l = ( x[1]*log(2*p[1] - p[1]^2) + x[2]*log(p[2]^2 + 2*p[2]*p[3]) +
          2*x[3]*log(p[3]) )
    return(l)
}

Q_prime = function(n,p){
    da = (2*n[1]+n[2]+n[3])/(p[1]) - (2*n[6]+n[3]+n[5])/(p[3])
    db = (2*n[4]+n[5]+n[2])/(p[2]) - (2*n[6]+n[3]+n[5])/(p[3])
    dQ = c(da,db)
    return(dQ)
}

Q_2prime = function(n,p){
    da2 = -(2*n[1]+n[2]+n[3])/(p[1]^2) - (2*n[6]+n[3]+n[5])/(p[3]^2)
    db2 = -(2*n[4]+n[5]+n[2])/(p[2]^2) - (2*n[6]+n[3]+n[5])/(p[3]^2)
    dab = -(2*n[6]+n[3]+n[5])/(p[3]^2)
    d2Q = matrix(c(da2,dab,dab,db2), nrow=2, byrow=TRUE)
    return(d2Q)
}

## Main
l_old = alle_l(x,p)
for(i in 1:iter){
    alpha = alpha_def
    n = alle_e(x,p)
    m = Q_2prime(n,p) - b
    p_new = p[1:2] - alpha*solve(m)%*%Q_prime(n,p)
    p_new[3] = 1 - p_new[1] - p_new[2]
    if(all(p_new > 0) && all(p_new < 1)){l_new = alle_l(x,p_new)}
    # REDUCE ALPHA UNTIL A CORRECT STEP IS REACHED
    while(any(p_new < 0) || any(p_new > 1) || l_new < l_old){
        alpha = alpha/2
        p_new = p[1:2] - alpha*solve(m)%*%Q_prime(n,p)
        p_new[3] = 1 - p_new[1] - p_new[2]
        if(all(p_new > 0) && all(p_new < 1)){l_new = alle_l(x,p_new)}
    }
    at = p_new[1:2]-p[1:2]
    n = alle_e(x,p_new)
    bt = Q_prime(n,p)-Q_prime(n,p_new)
    vt = bt - b%*%at
    ct = as.numeric(1/(t(vt)%*%at))
    b = b + ct*vt%*%t(vt)
    p = p_new
    prob_vals[,i+1] = p
    l_old = l_new
}

## Output
p   # Final Estimate for Allele Probabilities (p_c, p_i, p_t)

## Plot of Convergence
pb = seq(0.001,0.4,length=100)
c = outer(pb,pb,function(a,b){1-a-b})
pbs = matrix(0,3,10000)
pbs[1,] = rep(pb,100)
pbs[2,] = rep(pb,each=100)
pbs[3,] = as.vector(c)
z = matrix(apply(pbs,2,alle_l,x=x),100,100)

contour(pb,pb,z,nlevels=20)
for(i in 1:iter){
    segments(prob_vals[1,i],prob_vals[2,i],prob_vals[1,i+1],
      prob_vals[2,i+1],lty=2)
}
#########################################################################
### End of File

