#########################################################################
# COMPUTATIONAL STATISTICS
# by Geof Givens and Jennifer Hoeting
# Chapter 6 Examples
#########################################################################

# pacman must already be installed; then load contributed
# packages (including pacman) with pacman
pacman::p_load(pacman, rio, tidyverse, VGAM)

#Set Working Directory to Source File Location
library("rstudioapi")                                 # Load rstudioapi package
setwd(dirname(getActiveDocumentContext()$path))       # Set working directory to source file location
getwd()                                               # Check updated working directory

#########################################################################
### EXAMPLE 6.1 GAMMA DEVIATES
#########################################################################
# r = gamma dist shape parameter (r >= 1)
# n = sample size
# g = function for generating gamma(r,1) draw
# x = random draws from gamma(r,1) dist
#########################################################################
# INITIAL VALUES
r <- 4
n <- 1000
z <- rnorm(n)
u <- runif(n)

# FUNCTIONS
# Use more descriptive variable names and avoid global variables inside functions
g_tr <- function(y, rate) {
  coeff <- rate - (1/3)
  scale_fac <- 1 / sqrt(9 * coeff)
  tr_out <- coeff * (1 + scale_fac * y)^3
  return(tr_out)
}

# Vectorizing the g_tr function for efficiency
g_tr_vec <- Vectorize(g_tr)

# MAIN
coeff <- r - (1/3)
v <- g_tr_vec(z, r)
valid_ind <- v > 0 #valid indices

# Use vector indexing instead of repeatedly updating z, u, and v
z_valid <- z[valid_ind]
u_valid <- u[valid_ind]
v_valid <- v[valid_ind]

# Calculation is done in a vectorized manner for efficiency
f <- exp(((z_valid^2) / 2) + coeff * log(v_valid / coeff) - v_valid + coeff)
keep_ind <- u_valid <= f

# OUTPUT
# Draws from Gamma(r,1) using the indices of kept draws
x <- v_valid[keep_ind]

# Printing the percent of draws accepted
acc_rate <- length(x) / n
print(acc_rate)

# PLOTS
# Setting up the plot margins once for all plots
par(mar = c(7, 5, 5, 1))

# Plot of Envelope
plot(sort(z), exp(-(sort(z)^2) / 2), type = "l", ylab = expression(paste("exp(-(", z^2, ")/2)")), 
     main = "Envelope and Density")

# Plot of Density proportional to target ignoring normalizing constant
lines(sort(z), exp(coeff * log(g_tr_vec(sort(z), r) / coeff) - g_tr_vec(sort(z), r) + coeff), col = 2, lty = 2)

#########################################################################
### EXAMPLE 6.2 SAMPLING A BAYESIAN POSTERIOR
#########################################################################
# x_obs = observed data
# n_sam = number of samples to draw
# prior_draws = draws from the prior
# unif_draws = uniform random draws
#########################################################################

# INITIAL VALUES
x_obs <- c(8, 3, 4, 3, 1, 7, 2, 6, 2, 7)
n_sam <- 1000
prior_draws <- exp(rnorm(n_sam, log(4), 0.5))
unif_draws <- runif(n_sam)
x_obs_bar <- mean(x_obs)

# MAIN
log_post_diff <- sapply(prior_draws, function(lambda) {
  sum(dpois(x_obs, lambda, log = TRUE)) - sum(dpois(x_obs, x_obs_bar, log = TRUE))
})
post_wts <- exp(log_post_diff)
acc_ind <- (1:n_sam)[unif_draws < post_wts]

# OUTPUT
post_draws <- prior_draws[acc_ind] # DRAWS FROM POSTERIOR
acc_rate <- length(post_draws) / n_sam
print(acc_rate)

# PLOTS
lam_range <- seq(0, 20, length.out = 1000)
unnorm_prior_dens <- dlnorm(lam_range, log(4), 0.5) * prod(dpois(x_obs, x_obs_bar))

plot(lam_range, unnorm_prior_dens * 10^11, type = "l", xlab = expression(lambda),
     ylab = expression(paste("Unnormalized Density x ", 10^11)), main = "Posterior Sampling Visualization")

unnorm_post_dens <- dlnorm(lam_range, log(4), 0.5) * sapply(lam_range, function(lambda) prod(dpois(x_obs, lambda)))
lines(lam_range, unnorm_post_dens * 10^11, lty = 2, col = 2)

legend("topright", legend = c("Envelope", "Unnormalized Target"),
       col = c("black", "red"), lty = 1:2, cex = 1)

#########################################################################
### EXAMPLE 6.3 SLASH DISTRIBUTION
#########################################################################
# m = sample size
# n = resample size
# y = sample candidates from a standard normal dist
# v = sample candidates from a slash dist
# w = computes standardized importance weights (slash target density)
# w2 = computes standardized importance weights (normal target density)
# x = resample (approximate draws from slash dist)
# u = resample (approximate draws from standard normal dist)
#########################################################################
## NOTES
# This example uses the {VGAM} package for its implementation of the
# slash distribution.
#########################################################################

# INITIAL VALUES
sam_size <- 100000
resam_size <- 5000
norm_cand <- rnorm(sam_size) #normal candidates
slash_cand <- rslash(sam_size) #slash candidates

# FUNCTIONS
# Computes standardized importance weights for the slash target density
comp_wts_slash <- function(x) {
  weights <- dslash(x) / dnorm(x)
  weights / sum(weights) # Return normalized weights
}

# Computes standardized importance weights for the normal target density
comp_wts_norm <- function(x) {
  weights <- dnorm(x) / dslash(x)
  weights / sum(weights) # Return normalized weights
}

# CALCULATE WEIGHTS AND RESAMPLE
slash_wts <- comp_wts_slash(norm_cand)
resam_slash <- sample(norm_cand, resam_size, replace = TRUE, prob = slash_wts)

norm_wts <- comp_wts_norm(slash_cand)
resam_norm <- sample(slash_cand, resam_size, replace = TRUE, prob = norm_wts)

## OUTPUT
# Setting up plotting area
par(mfrow = c(1, 2))

# Histogram and density for normal approximation
hist(resam_norm, freq = FALSE, breaks = seq(-7, 7, by = .25), main = "Normal Approximation", xlab = "x", ylab = "Density")
curve(dnorm, col = "red", add = TRUE, lwd = 2)

# Histogram and density for slash approximation
hist(resam_slash, freq = FALSE, breaks = seq(-7, 7, by = .25), main = "Slash Approximation", xlab = "x", ylab = "Density")
curve(dslash, col = "red", add = TRUE, lwd = 2)

par(mfrow = c(1, 1))
# Comparison plot
x_vals <- seq(-10, 10, by = .01)
plot(x_vals, dnorm(x_vals), type = "l", xlab = "x", ylab = "Density", main = "Standard Normal vs. Slash Density")
lines(x_vals, dslash(x_vals), col = "red", lty = 2)
legend("topright", legend = c("Standard Normal", "Slash"), col = c("black", "red"), lty = 1:2, cex = 1)

#########################################################################
### EXAMPLE 6.5 SIMPLE MARKOV PROCESS
#########################################################################
# tmax = limit on t index
# n = number of Monte Carlo samples
# rejuvs = number of rejuvenations used
# ess = effective sample size
# w = importance weights
# X = samples
# Xt = current sample of X_t | X_(t-1)
# sds = sequence of estimates of sigma
#########################################################################

# INITIAL VALUES
tmax <- 100
n <- 100000
X <- matrix(0, n, tmax + 1)
w <- matrix(0, n, tmax + 1)
w[, 1] <- rep(1 / n, n)
ess <- rep(0, tmax)
rejuvs <- 0

# MAIN LOOP
set.seed(919191)
for (t in 1:tmax) {
  Xt <- rnorm(n, X[, t], 1.5)
  imp_ratio <- abs(cos(Xt - X[, t])) * exp(-0.25 * abs(Xt - X[, t])^2) / dnorm(Xt, X[, t], 1.5)
  w[, t + 1] <- w[, t] * imp_ratio
  w[, t + 1] <- w[, t + 1] / sum(w[, t + 1])
  ess[t] <- 1 / sum(w[, t + 1]^2)
  
  # Rejuvenation
  if (ess[t] < 0.33 * n) {
    Xt <- sample(Xt, n, prob = w[, t + 1], replace = TRUE)
    rejuvs <- rejuvs + 1
    w[, t + 1] <- rep(1 / n, n)
  }
  
  X[, t + 1] <- Xt
}

# CALCULATE SIGMA FOR EACH T
sds <- rep(0, tmax + 1)
for (i in 1:(tmax + 1)) {
  mu_hat <- sum(w[, i] * X[, i]) / sum(w[, i])
  sum_of_sq <- sum(w[, i] * (X[, i] - mu_hat)^2)
  sds[i] <- sqrt((1 / (1 - sum(w[, i]^2))) * sum_of_sq)
}

## OUTPUT
sig_hat_100 <- sds[101]  # SigmaHat at t=100
sig_hat_100

#########################################################################
### EXAMPLE 6.6 HIGH DIMENSIONAL DISTRIBUTION
#########################################################################
# p = number of dimensions
# n = number of points sampled
# N_eff = effective sample size
# X = sample
# w = importance weights
# rejuvs = number of rejuvenations
# norm_draw = sample from the standard normal envelope.
# ftxt = f_x(x_1:t) in book notation
# ft1xt1= f_t-1(x_1:t-1)
# Note: f_t(x_t|x_1:t-1) = ftxt/ft1xt1
#########################################################################

##INITIAL VALUES
p=50
n=5000
X=matrix(NA,nrow=n,ncol=p)
w=rep(1,n)/n
N_eff=rep(NA,p+1)
N_eff[1]=n
ft1xt1=w
rejuvs=0

##FUNCTION
dens_fnc=function(x) {
  temp=function(x) {   exp(-(abs(sqrt(sum(x^2)))^3)/3) }
  c(apply(x,1,temp)) }

##MAIN LOOP
set.seed(4567)
for (i in 1:p) {
    norm_draw=rnorm(n,mean=0,sd=1)
    if (i>1) {
         Xtemp=cbind(X[,1:(i-1)],norm_draw) 
         ftxt=dens_fnc(Xtemp) } else { ftxt=dens_fnc(cbind(norm_draw)) }
    X[,i]=norm_draw 
    w=w*ftxt/(ft1xt1*dnorm(norm_draw))
    ft1xt1=ftxt
    w=w/sum(w)
    N_eff[i+1]=1/sum(w^2)
    if (N_eff[i+1]<N_eff[1]/5) {
       rejuvs=rejuvs+1
       idx=sample(1:n,n,replace=T,prob=w)
       X=X[idx,]
       ft1xt1=dens_fnc(X[,1:i])
       w=rep(1,n)/n
    } 
}

#OUTPUT
rejuvs     # NUMBER OF REJUVENATIONS
N_eff      # EFFECTIVE SAMPLE SIZES AT EACH ITERATION
median(N_eff)

#NOW TRY TO EXPEND ALL EFFORT ON ONE SIR OF 5000 PTS
set.seed(111)
x=matrix(rnorm(50*5000),5000,50)
g=dnorm(x)
f=dens_fnc(x)
impwt=f/apply(g,1,prod)
impwt=impwt/sum(impwt)
#ESS
1/sum(impwt^2)  #Answer varies with random number seed

#########################################################################
### EXAMPLE 6.7 - 6.8  TERRAIN NAVIGATION
#########################################################################
# 
#########################################################################
#Note: This example is rather complex and less transparent than
#most examples we provide.  It will also be slow on your computer.
#########################################################################
# n = number of sampled trajectories
# Yt = observed elevation data
# mxti = map elevations
# xt_x, xt_y = current position of point
# uti = weight adjustment factors
# wti = weights
# Neff = effective sample size
# alpha = rejuvenation trigger
# dsubt_x, dsubt_y = true drift
# epst_x, epst_y = location error
#######################################################
# Note: this example requires the packages {akima} and {mvtnorm}
#######################################################
##The Colorado data are stored as a matrix suitable
##for reading and plotting using the image() command.
##The first row and column of the matrix are *not* 
##elevations.  They are lat,lon coordinates (with a NA
##in the 1,1 spot in the matrix.  The first row and column
##should therefore be stripped away (see below).  The
##remainder of the matrix gives elevations over the
##coordinate system.  In summary, the following R
##code should be used to read and arrange the data:
#######################################################
colo=as.matrix(read.table(file="../Datasets/colorado.dat",sep="",header=F))
colorado.lat=colo[1,-1]
colorado.lon=colo[-1,1]
colorado_elev=colo[-1,][,-1]
dim(colorado_elev)  #should be 186 x 125
image(colorado.lon,colorado.lat,colorado_elev,col=gray(seq(.25,1,len=200)),
 xlim=c(-6000,34000),ylim=c(-6000,34000))
####################################################

library(mvtnorm)
library(akima)

##INITIAL VALUES
n=100
sigma=75 
qval=400
k=.5 
alpha=0.3
sdx0=sqrtP0=50
wti=rep(1/n,n)
Neff_record=rep(NA,100)
rejuv_cnt=0
reset=F

##SET UP TRUE STARTING POINT, 100 INITIAL POINTS
##AND FIRST ELEVATION OBSERVATION
x0hat_x=0 ; x0hat_y=30000  #start at true X0 here
truex=x0hat_x
truey=x0hat_y
xthat_x=truex
xthat_y=truey
xt_x=rnorm(n,x0hat_x,sqrtP0)   #was x_xold 
xt_y=rnorm(n,x0hat_y,sqrtP0)   #was y_yold
colo_elev_interp=colorado_elev
colo_elev_interp[is.na(colo_elev_interp)]=
    mean(colo_elev_interp[!is.na(colo_elev_interp)])
colo_lon_rep=rep(colorado.lon,rep(125,186))
colo_lat_rep=rep(colorado.lat,186)
Yt=interp(colo_lon_rep,colo_lat_rep,
     colo_elev_interp,xo=0,yo=30000)$z

##SET UP TRUE DRIFT
route.theta=seq(0,pi/2,len=101)
route_x=30000*cos(route.theta)
route_y=30000*sin(route.theta)
route_x=rev(route_x)
route_y=rev(route_y)
dsubt_x=diff(route_x)
dsubt_y=diff(route_y)

##FUNCTION
rot_norm=function(N,slope,sigmamat) {
     v=rmvnorm(N,mean=c(0,0),sigma=sigmamat)
     xy=c(1,slope)
     Rot=cbind(c(-xy[1],-xy[2]),c(xy[2],-xy[1]))/sqrt(sum(xy^2))
     therot=t(Rot%*%t(v))
     therot }

image(colorado.lon,colorado.lat,colorado_elev,
      col=gray(seq(.25,1,len=200)),xlab="longitude (east-west)",ylab="latitude (south-north)",
               main="Flight Trajectory of an Airplane together with\
      Ground Elevations for a Region in Colorado", xlim=c(-6000,34000),ylim=c(-6000,34000))
lines(route_x,route_y,col=2,lwd=2) #truth

set.seed(75339)
for (i in 1:100) {
  #find m(x_t^i)
    xord=rank(xt_x)  
    yord=rank(xt_y)
    mapt=interp(colo_lon_rep,colo_lat_rep,colo_elev_interp,
       xo=sort(xt_x),yo=sort(xt_y))  
    mxti=mapt$z[cbind(xord,yord)] 
    extrap=is.na(mxti)
  #weight the points
    uti=ifelse(extrap,0,dnorm(Yt,c(mxti),rep(sigma,n)))
    wti=uti*wti
    wti=wti/sum(wti)
    Neff=1/sum(wti^2)
    Neff_record[i]=Neff
  #preliminary calcs for drawing the plot
    xthat.old=c(xthat_x,xthat_y)
    xthat_x=sum(wti*xt_x)
    xthat_y=sum(wti*xt_y)
  #check if resample needed
    if (Neff<(alpha*n)) {
       idx=sample(1:n,n,replace=T,prob=wti)
       xtnew_x=xt_x[idx]
       xtnew_y=xt_y[idx]
       wti=rep(1/n,n) 
       rejuv_cnt=rejuv_cnt+1
       reset=T }
  #update cloud
    tan_slope=-truex/truey
    Zsigmamat=cbind(c(qval^2,0),1*c(0,(k*qval)^2))
    xtnext=rot_norm(n,tan_slope,Zsigmamat)  
    epst_x=xtnext[,1]
    epst_y=xtnext[,2]
    if (!reset) {
       xtnext_x=xt_x+dsubt_x[i]+epst_x      #still using old points
       xtnext_y=xt_y+dsubt_y[i]+epst_y 
    } else {
       xtnext_x=xtnew_x+dsubt_x[i]+epst_x   #start with new points
       xtnext_y=xtnew_y+dsubt_y[i]+epst_y 
       reset=F
    }
    xt_x=xtnext_x
    xt_y=xtnext_y
  #update truth and observed elevation data
    truex=truex+dsubt_x[i]
    truey=truey+dsubt_y[i]
    Yt=interp(colo_lon_rep,colo_lat_rep,
         colo_elev_interp,xo=truex,yo=truey)$z+rnorm(1,0,sigma) 
     lines(c(xthat.old[1],xthat_x),
               c(xthat.old[2],xthat_y),lwd=2) 
}

##OUTPUT  
             #Graph is made above
rejuv_cnt   #Number of rejuvenations

#Note, for bootstrap filter, we force the resampling step 
#to occur at each iteration, and hence we also set wi = 1 
#anew at each iteration. There is no multiplicative
#adjustment to the previous stage's weight.

#########################################################################
### EXAMPLE 6.9 NETWORK FAILURE
#########################################################################
# n = sample size
# p = failure probability
# pstar = importance sampling method failure probability
# numedges = number of edges in complete network
#########################################################################

##INITIAL VALUES
p=.05
pstar=.2
n=100000
numedges=20
failed.orig=rep(FALSE,n)
fail_IS=rep(FALSE,n)    

##SETTING UP THE NETWORK
#PRE-PROCESSING NOT SHOWN HERE GENERATED A LIST OF ALL 
#CONNECTED PATHS.  THIS SHORTCUT IS USED TO SPEED THE EXAMPLE.
#Letters indicate links.  Characters assigned alphabetically
#working from left to right and then top to down.  Thus
#a, b, c, d represent the four links from A to one of the
#nodes directly to it's right.  The top node in the
#second column has links e, f, g, h to the four nodes
#in the third column.  Then labels are given to the second node 
#in the second column.  And so forth.

connecteds=c("ahnr","ahnos","ahnopt","ahnolmt","ahnokgmt",
  "ahnjfks","ahnjfkpt","ahnjfklmt","ahnjfgls","ahnjfglpt",
  "ahnjfgmt","ahnjfgmps","ahq","ainq","air","aios","aiopt",
  "aiolmt","aiokfmt","aijfgls","aijfglpt","aijfgmt","aijfgmps",
  "aijfks","aijfkpt","aijfklmt","aejnq","aejr","aejos",
  "aejopt","aejolmt","aejokgmt","aefkonq","aefkor","aefks",
  "aefkpt","aefklmt","aefglonq","aefgls","aefglpt","aefgmt",
  "aefgmps","aefgmpor","aefgmponq","behq","behnr","behnos",
  "behnopt","behnokgmt","behnolmt","beinq","beir","beios",
  "beiopt","beiolmt","beiokgmt","bjnq","bjr","bjos","bjopt",
  "bjolmt","bjokgmt","bjihq","bfkonq","bfkor","bfkoihq",
  "bfks","bfkpt","bfklmt","bfgloihq","bfglonq","bfglor",
  "bfgls","bfglpt","bfgmps","bfgmpor","bfgmpoihq","bfgmponq",
  "bfgmt","cfehq","cfehnr","cfehnos","cfehnolmt","cfehnopt",
  "cfeinq","cfeir","cfeios","cfeiopt","cfeiolmt","cfjnq",
  "cfjr","cfjos","cfjopt","cfjolmt","cfjihq","ckonq","ckor",
  "ckojehq","ckoihq","cks","ckpt","cklmt","cglonq","cglor",
  "cglojehq","cgloihq","cgls","cglpt","cgmponq","cgmpor",
  "cgmpojehq","cgmpoihq","cgmps","cgmt","dgfehq","dgfehnr",
  "dgfehnos","dgfehnopt","dgfeinq","dgfeir","dgfeios",
  "dgfeiopt","dgfjnq","dgfjr","dgfjos","dgfjopt","dgfjihq",
  "dgkonq","dgkor","dgkojehq","dgks","dgkpt","dlonq","dlor",
  "dlojehq","dloihq","dls","dlpt","dlkfehq","dlkfehnr",
  "dlkfeir","dlkfeinq","dlkfjr","dlkfjnq","dlkfjihq","dmponq",
  "dmpor","dmpojehq","dmpoihq","dmps","dmpkfehq","dmpkfehnr",
  "dmpkfeinq","dmpkfeir","dmpkfjihq","dmpkfjnq","dmpkfjr","dmt")

##GENERATE LOGICAL MATRIX STRUCTURE FLAGGING NECESSARY LINKS
##IN EACH POSSIBLE LINKED CONFIGURATION	
temp=matrix(0,nrow=158,ncol=9)
for (i in 1:9) {
   temp[,i]=substring(connecteds,i,i) 
}

mylet=c("",letters[1:numedges])
linkmat=matrix(0,158,9)
for (i in 1:9) {
  for (j in 1:158) {
    linkmat[j,i]=(0:numedges)[mylet==temp[j,i]]
  }
}

link.logic.mat=matrix(F,nrow=158,ncol=numedges)
for (i in 1:158) {
     for (j in 1:9) {
          if (linkmat[i,j]>0) {link.logic.mat[i,linkmat[i,j]]=TRUE} 
     } 
}

##CORE OF EXAMPLE BEGINS HERE

##SAMPLE NETWORKS AND CHECK WHETHER THEY FAIL
set.seed(126)
breaks=sample(c(T,F),numedges*n,prob=c(p,1-p),replace=T)
brokenlinks=matrix(breaks,nrow=n,ncol=numedges)
for (j in 1:n) {
config=brokenlinks[j,]
     if (sum(config)>0) {
          tested=t( t(link.logic.mat)&(!config) )
          failed.orig[j]=!any(!apply(xor(tested,link.logic.mat),1,any)) 
     } 
}

##IMPORTANCE SAMPLING APPROACH
set.seed(127)
breaks=sample(c(T,F),numedges*n,prob=c(pstar,1-pstar),replace=T)
brokenlinks=matrix(breaks,nrow=n,ncol=numedges)

for (j in 1:n) {
    config=brokenlinks[j,]
    if (sum(config)>0) {
        tested=t( t(link.logic.mat)&(!config) )
        fail_IS[j]=!any(!apply(xor(tested,link.logic.mat),1,any)) 
    } 
}

fail_in_conf=apply(brokenlinks,1,sum)
weights=fail_in_conf*(log(p)-log(pstar))+
         (numedges-fail_in_conf)*(log(1-p)-log(1-pstar))
weights=exp(weights)

##OUTPUT
sum(failed.orig)                #original count of failed networks
sum(fail_IS)                  #count of failed networks with pstar=.25
mean(fail_IS*weights)         #importance sampling estimate muhatstar_is
sqrt(var(fail_IS*weights)/n)  #MC standard error

#########################################################################
### EXAMPLE 6.10 NORMAL EXPECTATION
#########################################################################
# n = sample size
# z = standard normal draws
# h = target function
#########################################################################

## INITIAL VALUES
n = 100000
z = rnorm(n)

## FUNCTIONS
h = function(x){x/((2^x)-1)}

## MONTE CARLO ESTIMATE
mu_mc = mean(h(z))
se_mc = sd(h(z))/sqrt(n)

## ANTITHETIC ESTIMATOR
mu_a = sum(h(z[1:50000])+h(-z[1:50000]))/n
rho = cor(h(z[1:50000]),h(-z[1:50000]))
se_a = (1+rho)*var(h(z[1:50000]))/n

## OUTPUT
mu_mc # MONTE CARLO ESTIMATOR
se_mc # MONTE CARLO STD ERROR
mu_a # ANTITHETIC ESTIMATOR
se_a # ANTITHETIC STD ERROR
rho # CORRELATION

#########################################################################
### EXAMPLE 6.13 OPTION PRICING
#########################################################################
# S = prices of stock over time
# S0 = price at time zero
# ST = price at time T
# K = strike price
# T = time of maturity
# r = risk-free rate of return
# n = number of iterations
# m = number of mc estimations
# sigma = stock's volatility
#########################################################################

## INITIAL VALUES
S0 = 100
K = 102
sigma = 0.3
T = 50
r = 0.05
n = 1000
m = 100

## BACKGROUND: MC ESTIMATES (EUROPEAN CALL OPTION)
mu_mc_e = NULL
for(j in 1:m){
      ST = S0*exp((r-(sigma^2)/2)*T/365 + sigma*rnorm(n)*sqrt(T/365))
      C = NULL
      for(i in 1:n){
      	    C[i] = exp(-r*T/365)*max(c(0,ST[i] - K))
      }
      mu_mc_e[j] = mean(C)
}
se_mc_e = sd(mu_mc_e)/sqrt(m)

## MC ESTIMATES (ASIAN ARITHMETIC AND GEOMETRIC CALL OPTION)
mu_mc = NULL
theta_mc=NULL
for(j in 1:m){
 #calculate MC estimate of A and theta
      A = NULL
      theta = NULL
      for(i in 1:n){
      	    ST = NULL
	    ST[1] = S0
	    for(k in 2:T){
	    	  ST[k] = ST[k-1]*exp(((r-(sigma^2)/2)/365) +
		    sigma*rnorm(1)/sqrt(365))
            }
      A[i] = exp(-r*T/365)*max(c(0,mean(ST) - K))
      theta[i] = exp(-r*T/365)*max(c(0,exp(mean(log(ST))) - K))
      }
      mu_mc[j] = mean(A)
      theta_mc[j]=mean(theta)
}

## ANALYTIC SOLUTION (GEOMETRIC MEAN)
N = T
c3 = 1 + 1/N
c2 = sigma*((c3*T/1095)*(1 + 1/(2*N)))^.5
c1 = (1/c2)*(log(S0/K) + (c3*T/730)*(r - (sigma^2)/2) +
       (c3*(sigma^2)*T/1095)*(1 + 1/(2*N)))
theta = S0*pnorm(c1)*exp(-T*(r + c3*(sigma^2)/6)*(1 - 1/N)/730) -
       K*pnorm(c1-c2)*exp(-r*T/365)

## CONTROL VARIATE
mu_cv=mu_mc-1*(theta_mc-theta)

## OUTPUT
sd(mu_mc)  #STANDARD DEVIATION FOR ORDINARY APPROACH
sd(mu_cv)  #STANDARD DEVIATION FOR CONTROL VARIATE APPROACH

#########################################################################
### END OF FILE

