#########################################################################
# COMPUTATIONAL STATISTICS
# by Geof Givens and Jennifer Hoeting
# CHAPTER 8 EXAMPLES
#########################################################################

# pacman must already be installed; then load contributed
# packages (including pacman) with pacman
pacman::p_load(pacman, rio, tidyverse, VGAM)

#Set Working Directory to Source File Location
library("rstudioapi")                                 # Load rstudioapi package

setwd(dirname(getActiveDocumentContext()$path))       # Set working directory to source file location
getwd()                                               # Check updated working directory

############################################################################
### EXAMPLE 8.1 WHALE POPULATION DYNAMICS (ADAPTIVE METROPOLIS-WITHIN-GIBBS)
############################################################################

# Variables:
# K=carrying capacity (see Equation 8.2)
# r=intrinsic growth rate
# N=abundance (population count)
# C=catch
# sigma=variance of log(N.hat) (see Equation 8.3)
# true_cv=coefficient of variation (see Equation 8.3)
# Xt = survey abundance estimate (this is called \hat{N}_y in Example 8.1)
# 
# Functions:
# get_traj=population trajectory
# myloglik.tomax=log likelihood used for maximum likelihood estimation (not MCMC)
# myloglik=log likelihood used for AMCMC
# calc_post= computes the value of the posterior
# mypost_tomax=posterior used to compute the MAP (maximum a posteriori) estimate
# trc_norm=generates from the proposal distribution, see equations 8.4 and 8.6
# trc_unif=generates from the proposal distribution, see equation 8.5

###########Generate the data
K=10000
r=.03
N=rep(0,101)
sigma=0.30
true_cv=sqrt(exp(sigma^2)-1)

#catch data
C=rep(0,101)
C=c(606,1151,928,1024,1375,939,701,573,350,322,278,254,
   rep(60,8),rep(30,10),rep(25,30),rep(20,41))

#Add noise to the catch data
set.seed(19359)
for (i in 13:101) {
  C[i]=C[i]*exp(rnorm(1,0,.5)) }
C=round(C,0)

#plot the true traj
N[1]=K
for(t in 1:100) {
  N[t+1]=N[t]-C[t]+r*N[t]*(1-(N[t]/K)^2) }
N=round(N)
plot(1:101,N,type="l",xlab="t",ylim=c(0,K+3000))

#use only surveys from five years
dat_years=c(14,21,63,91,93,100)
n=length(dat_years)

#generate surveys
set.seed(477)
Xt=N*exp(rnorm(101,0,sigma))
Xt=round(Xt,0)
#points(1:101,Xt)
points(dat_years,Xt[dat_years],pch=16,col=2)

## FUNCTIONS

get_traj=function(theK,ther) {
  Ntraj=rep(NA,101)
  Ntraj[1]=theK
  for(t in 1:100) {
    Ntraj[t+1]=Ntraj[t]-C[t]+ther*Ntraj[t]*(1-(Ntraj[t]/theK)^2) 
    if (Ntraj[t+1]<=0) { Ntraj[(t+1):101]=0.0001 ; break } }
    Ntraj 
}

myloglik.tomax=function(parms,dy=dat_years) {
  K=parms[1]
  r=parms[2]
  cv=parms[3]
  sigma=sqrt(log(1+cv^2))
  Ntraj=get_traj(K,r) 
  nll=-sum(log(dlnorm(Xt[dy],log(Ntraj[dy]),sigma)))
  if(is.infinite(nll)) {nll=1e10}  
  nll }
 

myloglik=function(K,r,cv,dy=dat_years) {
  sigma=sqrt(log(1+cv^2))
  Ntraj=get_traj(K,r) 
  sum(log(dlnorm(Xt[dy],log(Ntraj[dy]),sigma))) } 

calc_post=function(K,r,cv) {
  ml=myloglik(K,r,cv) 
  if (is.infinite(ml) | K<7000 | K>100000 | r<.001 | r>.1 | cv<0 | cv>2) {p=0} else {
    pripart=(1/93000)*(1/.099)*dbeta(cv/2,2,10) 
    p=exp(ml)*pripart }
  p }

#functions needed to calculate transition probabilities

trc_norm=function(qlow=-Inf,qhi=Inf,current,sd) {
  plow=pnorm(qlow,current,sd)
  phi=pnorm(qhi,current,sd)
  u=runif(1,plow,phi)
  next_draw=qnorm(u,current,sd)
  dfwd=dnorm(next_draw,current,sd)/(phi-plow)
  plow=pnorm(qlow,next_draw,sd)
  phi=pnorm(qhi,next_draw,sd)
  dbwd=dnorm(current,next_draw,sd)/(phi-plow)
  list(next_draw=next_draw,dfwd=dfwd,dbwd=dbwd) }

trc_unif=function(qlow,qhi,current,lim) {
  lower=max(qlow,current-lim)
  upper=min(qhi,current+lim)
  next_draw=runif(1,lower,upper)
  dfwd=1/(upper-lower)
  lower=max(qlow,next_draw-lim)
  upper=min(qhi,next_draw+lim)
  dbwd=1/(upper-lower)
  list(next_draw=next_draw,dfwd=dfwd,dbwd=dbwd) }

## MAIN

#Initialize

chn_len=45000 ; burnin=10000
post_prob=rep(NA,chn_len)
trigger=1500  #how often to adapt
Kpath = rpath = cvpath = rep(NA,chn_len+1)
Kpath[1]=10000 ; rpath[1]=.03 ; cvpath[1]=.3
mh_rat_chk_K = mh_rat_chk_r = mh_rat_chk_cv = rep(NA,chn_len)
as_jd_K = as_jd_r = as_jd_cv = rep(NA,chn_len)
the_fact_K = the_fact_r = the_fact_cv = rep(NA,chn_len)
acc_cnt_K = acc_cnt_r = acc_cnt_cv =0
acc_rat_K = acc_rat_r = acc_rat_cv = rep(NA,chn_len/trigger)

#first good run: new_K_sd= c(500,rep(NA,chn_len/trigger-1)) #results1
#new_K_sd= c(50,rep(NA,chn_len/trigger-1))     #results2
new_K_sd= c(200,rep(NA,chn_len/trigger-1))     #results3 (book choice)
new_r_lim= c(.03,rep(NA,chn_len/trigger-1))
new_cv_sd= c(.1,rep(NA,chn_len/trigger-1))
set_pnt=1

set.seed(123)
for (t in 1:(chn_len)) {  
#browser()  
  ##adapt if necessary
 if (t>1 & (t-1)%%trigger==0) {   
    set_pnt=set_pnt+1
  #for K
    acc_rat_K[set_pnt]=acc_cnt_K/trigger
    acc_cnt_K=0  
    upordown=ifelse(acc_rat_K[set_pnt]<.44,-1,1)
    the_fact_K[set_pnt] = exp(upordown*t^(-1/3))
    new_K_sd[set_pnt]=new_K_sd[set_pnt-1]*the_fact_K[set_pnt] 
    new_K_sd[set_pnt]=min(10000,new_K_sd[set_pnt]) 
  #for r
    acc_rat_r[set_pnt]=acc_cnt_r/trigger
    acc_cnt_r=0  
    upordown=ifelse(acc_rat_r[set_pnt]<.44,-1,1)
    the_fact_r[set_pnt] = exp(upordown*t^(-1/3))
    new_r_lim[set_pnt]=new_r_lim[set_pnt-1]*the_fact_r[set_pnt]
    new_r_lim[set_pnt]=min(.05,new_r_lim[set_pnt]) 
  #for cv
    acc_rat_cv[set_pnt]=acc_cnt_cv/trigger
    acc_cnt_cv=0  
    upordown=ifelse(acc_rat_cv[set_pnt]<.44,-1,1)
    the_fact_cv[set_pnt] = exp(upordown*t^(-1/3))
    new_cv_sd[set_pnt]=new_cv_sd[set_pnt-1]*the_fact_cv[set_pnt] 
    new_cv_sd[set_pnt]=min(10,new_cv_sd[set_pnt]) }

  ##proposals
  gk=trc_norm(7000,100000,Kpath[t],new_K_sd[set_pnt])
  prop.K=gk$next_draw
  gr=trc_unif(.001,.1,rpath[t],new_r_lim[set_pnt])
  prop.r=gr$next_draw
  gcv=trc_norm(0,2,cvpath[t],new_cv_sd[set_pnt])
  prop.cv=gcv$next_draw
  ##Gibbs cycles
   #for K
 mhr=calc_post(prop.K,rpath[t],cvpath[t])*gk$dbwd/
        (calc_post(Kpath[t],rpath[t],cvpath[t])*gk$dfwd)
 mh_rat_chk_K[t]=mhr
 as_jd_K[t]=min(mhr,1)*(prop.K-Kpath[t])^2
 u=runif(1,0,1)
 if (mhr>=1 | u<=mhr) {
   acc_cnt_K=acc_cnt_K+1
   Kpath[t+1]=prop.K } else {
   Kpath[t+1]=Kpath[t] } 
  #for r
#if (t>120) browser()
 mhr=calc_post(Kpath[t+1],prop.r,cvpath[t])*gr$dbwd/
        (calc_post(Kpath[t+1],rpath[t],cvpath[t])*gr$dfwd)
 mh_rat_chk_r[t]=mhr
 as_jd_r[t]=min(mhr,1)*(prop.r-rpath[t])^2
 u=runif(1,0,1)
 if (mhr>=1 | u<=mhr) {
   acc_cnt_r=acc_cnt_r+1
   rpath[t+1]=prop.r } else {
   rpath[t+1]=rpath[t] } 
 #for cv
 mhr=calc_post(Kpath[t+1],rpath[t+1],prop.cv)*gcv$dbwd/
        (calc_post(Kpath[t+1],rpath[t+1],cvpath[t])*gcv$dfwd)
 mh_rat_chk_cv[t]=mhr
 as_jd_cv[t]=min(mhr,1)*(prop.cv-cvpath[t])^2
 u=runif(1,0,1)
 if (mhr>=1 | u<=mhr) {
   acc_cnt_cv=acc_cnt_cv+1
   cvpath[t+1]=prop.cv } else {
   cvpath[t+1]=cvpath[t] }   
 post_prob[t]=calc_post(Kpath[t+1],rpath[t+1],cvpath[t+1])
}

## OUTPUT

plot(1:(chn_len+1),Kpath,type="l",xlab="t",ylab="K")
who=ifelse(mh_rat_chk_K==0,T,F)
points((2:chn_len+1)[who],Kpath[2:chn_len][who],pch=16,col=2)
new_K_sd
acc_rat_K
m1=mean(Kpath[burnin:chn_len])  #true K = 10000
m1
#[1] 9680 (Note:  your answers will vary)
summary(Kpath[burnin:chn_len])

hist(Kpath[burnin:chn_len],nclass=50,xlab="K",main="Histogram of K")
#mode 9748

plot(1:(chn_len+1),rpath,type="l",xlab="t",ylab="r")
who=ifelse(mh_rat_chk_r==0,T,F)
#text(2:(chn_len+1),rpath[2:chn_len],round(mh_rat_chk_r,4),col=who,cex=.75)
points((2:chn_len+1)[who],Kpath[2:chn_len][who],pch=16,col=2)
new_r_lim
acc_rat_r
m2=mean(rpath[burnin:chn_len])   #true r = 0.03
m2 
#[1] .04103 (Note:  your answers will vary)

summary(rpath[burnin:chn_len])

hist(rpath[burnin:chn_len],nclass=50,xlab="r",main="Histogram of r")
#.03295 mode

plot(1:(chn_len+1),cvpath,type="l",xlab="t",ylab="CV")
who=ifelse(mh_rat_chk_cv==0,T,F)
#text(2:(chn_len+1),cvpath[2:chn_len],round(mh_rat_chk_cv,4),col=who,cex=.75)
points((2:chn_len+1)[who],Kpath[2:chn_len][who],pch=16,col=2)
new_cv_sd
acc_rat_cv
m3=mean(cvpath[burnin:chn_len]) #true cv= 0.307
m3 
#[1] 0.2509253
summary(cvpath[burnin:chn_len])
hist(cvpath[burnin:chn_len],nclass=50,xlab="CV",main="Histogram of CV")
#mode 
#[1] .16295

#plot the true traj
N[1]=K
for(t in 1:100) {
  N[t+1]=N[t]-C[t]+r*N[t]*(1-(N[t]/K)^2) }
plot(1:101,N,type="l",ylim=c(0,K+3000),xlab="t")
points(dat_years,Xt[dat_years],pch=16,col=2)

#plot MLE
optim(c(14000,.04,.3),myloglik.tomax,
  control=list(parscale=c(10000,1/100,1/10))) 

optim(c(10000,.03,.3),myloglik.tomax)
#max is
#[1] 9.906669e+03 3.167191e-02 1.258889e-01

#plot MLE
Ntemp=rep(NA,101)
Ntemp[1]=  9907
rtemp=   .03167191
h=get_traj(Ntemp[1],rtemp)
lines(1:101,h,type="l",col=2,lwd=2)

#MAP estimate
mypost_tomax=function(params) {
  -log(calc_post(params[1],params[2],params[3])) }

optim(c(12000,.04,.3),mypost_tomax, 
  control=list(trace=2,parscale=c(10000,1/100,1/10)))

#max at
#[1] 9.906014e+03 3.168095e-02 1.300854e-01


#posterior

#priors:
#K ~ unif(7000,100000)
#r ~ unif(.001,.1)
#cv/2 ~ beta(2,10)  #meancv=.336

Ntemp=rep(NA,101)
Ntemp[1]=  9888
rtemp=   .03226434
h=get_traj(Ntemp[1],rtemp)
lines(1:101,h,type="l",col=2,lwd=2)

h=get_traj(9748,.03295)
lines(1:101,h,col=4)

who=post_prob==max(post_prob)  #joint MAP
cbind(Kpath,rpath,cvpath)[-1,][who,]
#       Kpath        rpath       cvpath 
#9.913550e+03 3.165349e-02 1.317438e-01 

h=get_traj(9914,.0316535)  #joint MAP
lines(1:101,h,col=6)

par(mfrow=c(1,2))
plot(1:30,acc_rat_K,type="l",xlab="t",ylab="accept ratio of K",ylim=c(0.20,0.68))
abline(h=0.44,lty=2)
plot(1:30,new_K_sd,type="l",xlab="t",ylab=expression(paste("200 ",delta["k"]^"t")))

par(mfrow=c(1,2))
plot(1:30,acc_rat_r,type="l",xlab="t",ylab="accept ratio of r",ylim=c(0.20,0.68))
abline(h=0.44,lty=2)
plot(1:30,new_r_lim,type="l",xlab="t",ylab=expression(paste("200 ",delta["r"]^"t")))

par(mfrow=c(1,2))
plot(1:30,acc_rat_cv,type="l",xlab="t",ylab="accept ratio of cv",ylim=c(0.20,0.68))
abline(h=0.44,lty=2)
plot(1:30,new_cv_sd,type="l",xlab="t",ylab=expression(paste("200 ",delta[psi]^"t")))

par(mfrow=c(1,1))
#create a list of all the relevant output
results3=list(Kpath=Kpath,rpath=rpath,cvpath=cvpath,
  acc_rat_K=acc_rat_K,acc_rat_r=acc_rat_r,
  acc_rat_cv=acc_rat_cv,new_K_sd=new_K_sd,new_r_lim=new_r_lim,
  new_cv_sd=new_cv_sd,as_jd_K=as_jd_K,as_jd_r=as_jd_r,as_jd_cv=as_jd_cv,
  post_prob=post_prob)

#only last 7500 iterations:
which=37500:45000
mean(as_jd_K[which])
mean(as_jd_r[which])
mean(as_jd_cv[which])

############################################################################
### EXAMPLE 8.3 BASEBALL SALARIES, CONTINUED (RJMCMC)
############################################################################
# baseball_dat = contains data
# ball_run1 = implements RJMCMC for the baseball data using the BMA package in R
############################################################################
## NOTES
#This example uses the BMA package in R.  You'll have to install the package
#first. 
#
#You should examine the MC3.REG algorithm from the BMA library in R so that you can 
#further understand the implementation of a fully working reversible jump MCMC algorithm.  
#The MC3.REG function calls several other functions including: 
#
# For.MC3.REG     = Helper function for MC3.REG which implements each step of 
#                   the Metropolis-Hastings algorithm.
# MC3.REG.logpost = Helper function to MC3.REG that calculates the posterior 
#                   model probability (up to a constant).
# MC3.REG.choose  = Helper function to MC3.REG that chooses the proposal model 
#                   for a Metropolis-Hastings step.
############################################################################

#INPUTS
library(BMA)  #install this package first
#baseball_dat = read.table(file.choose(),header=T)
baseball_dat = read.table(file="../Datasets/baseball.dat",sep="",header=T)

## MAIN
ball_run1<-MC3.REG(baseball_dat[,1], baseball_dat[,-1], num.its=200000, rep(TRUE,27), outliers = FALSE)

## OUTPUTS
summary(ball_run1)
ball_run1[1:20,]
ball_run1[200000,]

############################################################################
### EXAMPLE 8.6 UTAH SERVICEBERRY DISTRIBUTION (GIBBS SAMPLER)
############################################################################
# utah_berry_dat = contains data
# x         = true presence- true absence (1=presence, 0=absence)
# y         = observed presence- observed absence (1=presence, 0=absence)
# a         = alpha parameter 
# b         = beta parameter
# itr       = number of cycles to run
# y2        = alternate coding of y (-1=presence, 1=absence)
# xt        = estimated Presence-Absence information
#             (1=presence, 0=absence)
# x2        = alternate coding of xt (-1=presence, 1=absence)
# xt_all    = xt for each cycle
# xt_p      = mean posterior estimates for each pixel
# f         = computes probability of setting pixel(pos) to 1 (presence)
# neighbors = finds indices of the neighbors to pixel(pos)
############################################################################
## NOTES
# The image function in R rotates the matrix 90 degrees counter-clockwise.
# See the ?image help page for more information.
############################################################################

## INITIAL VALUES
#utah_berry_dat = read.csv(file.choose(),header=T)
utah_berry_dat = read.table(file="../Datasets/serviceberry.dat",sep="",header=T)
dim(utah_berry_dat)   
#Check dimensions to make sure that you have read in data correctly
#[1] 2484    1

head(utah_berry_dat)  
x = utah_berry_dat[,1]
a = 1
b = 0.8
itr = 100
set.seed(0)
y = x
n_row = 54
n_col = 46
n = n_row*n_col
bad = sample.int(n,745)
y[bad] = !y[bad]
y2 = (!y) - y
x2 = y2
xt = y
xt_all = matrix(0,itr,n)

## FUNCTIONS
f = function(pos){
      (1 + exp(a*y2[pos] + b*sum(x2[neighbors(pos)])))^-1
}

neighbors = function(pos){
      if((pos-1)%%n_row != 0 & pos%%n_row != 0){
            out = c(pos-1,pos+1,pos-n_row,pos+n_row)
            out = out[out>0 & out<(n+1)]
      }
      if((pos-1)%%n_row == 0 | pos%%n_row == 0){
            if(pos%%n_row != 0){
                  out = c(pos+1,pos-n_row,pos+n_row)
                  out = out[out>0 & out<(n+1)]
            }
            if(pos%%n_row == 0){
                  out = c(pos-1,pos-n_row,pos+n_row)
                  out = out[out>0 & out<(n+1)]
            }
      }
      return(out)
}

## MAIN
for(j in 1:itr){
      for(i in 1:n){
            q = f(i)
            d = rbinom(1,1,q)
            xt[i] = d
            x2[i] = 1-2*d
      }
      xt_all[j,] = xt
}

xt_p = colMeans(xt_all)
x_est = (xt_p>=.5)
p_true = 1-mean(abs(x_est - x))

## OUTPUT
# PERCENT OF CORRECT ESTIMATES 
p_true # POSTERIOR MEAN >= .5 IS CLASSIFIED AS 1, ELSE 0

## PLOTS
par(pty="s",mfrow=c(2,2))
x_image = matrix(x,n_row,n_col)
x_image = x_image[,n_col:1]
image(1:n_row,1:n_col,x_image, xlab="", ylab="", xaxt="n", yaxt="n",
      main="True Presence-Absence")

y_image = matrix(y,n_row,n_col)
y_image = y_image[,n_col:1]
image(1:n_row,1:n_col,y_image, xlab="", ylab="", xaxt="n", yaxt="n",
      main="Obs Presence-Absence")

xt_p_image = matrix(xt_p,n_row,n_col)
xt_p_image = xt_p_image[,n_col:1]
image(1:n_row,1:n_col,xt_p_image, xlab="", ylab="", xaxt="n", yaxt="n",
      main="Est Presence-Absence")

############################################################################
### EXAMPLE 8.7 UTAH SERVICEBERRY DISTRIBUTION (SWENDSEN-WANG)
############################################################################
# utah_berry_dat = contains data
# x         = true Presence-Absence information
# y         = observed Presence-Absence information
# a         = alpha parameter 
# b         = beta parameter
# itr       = number of cycles to run
# xt        = estimated Presence-Absence information (1=presence)
# xt_m      = matrix form of xt
# xt_all    = xt for each cycle
# xt_p      = mean posterior estimates for each pixel
# g         = computes probability of setting cluster to 1 (presence)
# neighbors = finds indices of the neighbors to pixel(pos)
# cluster   = finds indices of all pixels in cluster containing pixel(pos)
############################################################################
## INITIAL VALUES
utah_berry_dat = read.table(file="../Datasets/serviceberry.dat",sep="",header=T)
x = utah_berry_dat[,1]
a = 1
b = 0.8
itr = 100
set.seed(0)
y = x
n_row = 54
n_col = 46
n = n_row*n_col
bad = sample.int(n,745)
y[bad] = !y[bad]
xt = y
xt_all = matrix(0,itr,n)

## FUNCTIONS
neighbors = function(pos){
      if((pos-1)%%n_row != 0 & pos%%n_row != 0){
            out = c(pos-1,pos+1,pos-n_row,pos+n_row)
            out = out[out>0 & out<(n+1)]
      }
      if((pos-1)%%n_row == 0 | pos%%n_row == 0){
            if(pos%%n_row != 0){
                  out = c(pos+1,pos-n_row,pos+n_row)
                  out = out[out>0 & out<(n+1)]
            }
            if(pos%%n_row == 0){
                  out = c(pos-1,pos-n_row,pos+n_row)
                  out = out[out>0 & out<(n+1)]
            }
      }
      return(out)
}

g = function(clust){
      exp(a*sum(y[clust]))/(exp(a*sum(y[clust]==0)) + exp(a*sum(y[clust])))
}

cluster = function(pos){
      done = FALSE
      clust = c(pos)
      m = 1
      while(!done){
            pos = clust[m]
            k = neighbors(pos)
            more = NULL
            k_l = sort(k[k<pos])
            k_u = sort(k[!(k %in% k_l)])
            if(length(k_u)==1 & k_u[1]==(pos+1)){
                  if(bv[pos]){more = c(more,k_u[1])}}
            if(length(k_u)==1 & k_u[1]!=(pos+1)){
                  if(bh[pos]){more = c(more,k_u[1])}}
            if(length(k_u)==2){
                  if(bv[pos]){more = c(more,k_u[1])}
                  if(bh[pos]){more = c(more,k_u[2])}}
            if(length(k_l)==1 & k_l[1]==(pos-1)){
                  if(bv[pos-1]){more = c(more,k_l[1])}}
            if(length(k_l)==1 & k_l[1]!=(pos-1)){
                  if(bh[k_l[1]]){more = c(more,k_l[1])}}
            if(length(k_l)==2){
                  if(bh[k_l[1]]){more = c(more,k_l[1])}
                  if(bv[pos-1]){more = c(more,k_l[2])}}            
            if(length(more)>0){clust = unique(c(clust,more))}
            if(length(clust)==m){done=TRUE}
            m = m+1
            }
      return(clust)
}

## MAIN
for(k in 1:itr){
      # BUILDS HORIZONTAL AND VERTICAL BONDS
      xt_m = matrix(xt,n_row,n_col)
      bondv = matrix(0,n_row,n_col)
      bondh = matrix(0,n_row,n_col)
      for(j in 1:n_col){
      for(i in 1:n_row){
            if(i%%n_row != 0)
            {bondv[i,j] = runif(1,0,exp(b*(xt_m[i,j]==xt_m[i+1,j])))}
            if(j%%n_col != 0)
            {bondh[i,j] = runif(1,0,exp(b*(xt_m[i,j]==xt_m[i,j+1])))}
      }
      }
      bondv = bondv>1
      bondh = bondh>1
      bv = as.vector(bondv)
      bh = as.vector(bondh)

      # ASSIGNS COLORS TO CLUSTERS
      clust_check = rep(1,n)
      for(pos in 1:n){
            if(clust_check[pos]){
            clust = cluster(pos)
            q = g(clust)
            d = rbinom(1,1,q)
            xt[clust] = d
            clust_check[clust]=0
            }
      }
      xt_all[k,] = xt
}

xt_p = colMeans(xt_all)
x_est = (xt_p>=.5)
p_true = 1-mean(abs(x_est - x))

## OUTPUT
# PERCENT OF CORRECT ESTIMATES 
p_true # POSTERIOR MEAN >= .5 IS CLASSIFIED AS 1, ELSE 0

## PLOTS
par(pty="s",mfrow=c(2,2))
x_image = matrix(x,n_row,n_col)
x_image = x_image[,n_col:1]
image(1:n_row,1:n_col,x_image, xlab="", ylab="", xaxt="n", yaxt="n",
      main="True Presence-Absence")

y_image = matrix(y,n_row,n_col)
y_image = y_image[,n_col:1]
image(1:n_row,1:n_col,y_image, xlab="", ylab="", xaxt="n", yaxt="n",
      main="Obs Presence-Absence")

xt_p_image = matrix(xt_p,n_row,n_col)
xt_p_image = xt_p_image[,n_col:1]
image(1:n_row,1:n_col,xt_p_image, xlab="", ylab="", xaxt="n", yaxt="n",
      main="Est Presence-Absence")

############################################################################
### EXAMPLE 8.9 UTAH SERVICEBERRY DISTRIBUTION (CFTP)
############################################################################
# utah_berry_dat = contains data
# x         = true Presence-Absence information
# y         = observed Presence-Absence information
#             (1=presence, 0=absence)
# a         = alpha parameter 
# b         = beta parameter
# y2        = alternate coding of y (-1=presence, 1=absence)
# xt_black  = estimated Presence-Absence information for max space
#             (1=presence, 0=absence)
# x2_black  = alternate coding of max space (-1=presence, 1=absence)
# xt_white  = estimated Presence-Absence information for min space
#             (1=presence, 0=absence)
# x2_white  = alternate coding of min space (-1=presence, 1=absence)
# tau       = time to reach stationary distribution of the chain
# f         = computes probability of setting pixel(pos) to 1 (presence)
# neighbors = finds indices of the neighbors to pixel(pos)
############################################################################
## INITIAL VALUES
utah_berry_dat = read.table(file="../Datasets/serviceberry.dat",sep="",header=T)
x = utah_berry_dat[,1]
a = 1
b = 0.8
set.seed(0)
y = x
n_row = 54
n_col = 46
n = n_row*n_col
bad = sample.int(n,745)
y[bad] = !y[bad]
y2 = (!y) - y
xt_black = rep(1,n)
xt_white = rep(0,n)
x2_black = rep(-1,n)
x2_white = rep(1,n)
done = FALSE
tau = 0

## FUNCTIONS
f = function(pos){
      k = neighbors(pos)
      out = NULL
      out[1] = (1 + exp(a*y2[pos] + b*sum(x2_black[k])))^-1
      out[2] = (1 + exp(a*y2[pos] + b*sum(x2_white[k])))^-1
      return(out)
}
neighbors = function(pos){
      if((pos-1)%%n_row != 0 & pos%%n_row != 0){
            out = c(pos-1,pos+1,pos-n_row,pos+n_row)
            out = out[out>0 & out<(n+1)]
      }
      if((pos-1)%%n_row == 0 | pos%%n_row == 0){
            if(pos%%n_row != 0){
                  out = c(pos+1,pos-n_row,pos+n_row)
                  out = out[out>0 & out<(n+1)]
            }
            if(pos%%n_row == 0){
                  out = c(pos-1,pos-n_row,pos+n_row)
                  out = out[out>0 & out<(n+1)]
            }
      }
      return(out)
}

## MAIN
while(!done){
      for(i in 1:n){
            q = f(i)
            d = runif(1,0,1) < q
            xt_black[i] = d[1]
            xt_white[i] = d[2]
            x2_black[i] = 1-2*d[1]
            x2_white[i] = 1-2*d[2] 
      }
      if(sum(xt_black == xt_white)==n){done = TRUE}
      tau = tau + 1
}

## OUTPUT
tau # TIME TO REACH STATIONARY DISTRIBUTION OF THE CHAIN

#percent correct 
#This is quite poor.  Students:  improve the algorithm to see if you can improve % correct
1-mean(abs(x_image-x))


## PLOTS
par(pty="s",mfrow=c(2,2))
x_image = matrix(x,n_row,n_col)
x_image = x_image[,n_col:1]
image(1:n_row,1:n_col,x_image, xlab="", ylab="", xaxt="n", yaxt="n",
      main="True Presence-Absence")

y_image = matrix(y,n_row,n_col)
y_image = y_image[,n_col:1]
image(1:n_row,1:n_col,y_image, xlab="", ylab="", xaxt="n", yaxt="n",
      main="Obs Presence-Absence")

xt_image = matrix(xt_black,n_row,n_col)
xt_image = xt_image[,n_col:1]
image(1:n_row,1:n_col,xt_image, xlab="", ylab="", xaxt="n", yaxt="n",
      main="Stationary Distribution")

############################################################################
### END OF FILE
