#================================================================================#
# Stat 7650: Computational Statistics -- Spring 2025                             #
# R code examples for quadrature/numerical integration                           #
# Based in part on Givens & Hoeting (Chap 5) and Lange (Chap 18)                 #
# Modified from code of R. Martin                                                #
#================================================================================#

#Set Working Directory to Source File Location
library("rstudioapi")                                 # Load rstudioapi package
setwd(dirname(getActiveDocumentContext()$path))       # Set working directory to source file location
#getwd()   

#some functions for the methods used in G&H
source("AuxFunctions.r")

#library for the tidyverse environment
library(tidyverse)

## Example: Simple check -- integral below = 1/3.
x <- seq(0, 1, len=100)
fx <- x**2
myint(fx, x)
f <- function(x) x**2
integrate(f, lower=0, upper=1)

############################################################################
## Example: Normal Mixtures
############################################################################

# Define parameters
mu1 <- 0
sigma1 <- sqrt(1)  # Standard deviation
mu2 <- 3
sigma2 <- sqrt(4)  # Standard deviation
pi <- 0.5  # Mixing proportion

# Define y range
y <- seq(-5, 10, length.out = 300)

# Compute individual normal densities
dens1 <- dnorm(y, mean = mu1, sd = sigma1)
dens2 <- dnorm(y, mean = mu2, sd = sigma2)

# Compute the mixture density
mixture_density <- pi * dens1 + (1 - pi) * dens2

# Create a data frame for plotting
df <- data.frame(
  y = rep(y, 3),
  density = c(dens1, dens2, mixture_density),
  Distribution = factor(rep(c("N(y; 0,1)", "N(y; 3,4)", "Mixture"), each = length(y)))
)

# Plot
ggplot(df, aes(x = y, y = density, color = Distribution, linetype = Distribution)) +
  geom_line(size = 1.2) +
  labs(title = "Mixture of Two Normal Distributions",
       x = "y", y = "Density") +
  theme_minimal() +
  scale_color_manual(values = c("blue", "red", "black")) +
  scale_linetype_manual(values = c("solid", "solid", "dashed")) +
  theme(legend.position = "top")

############################################################################
## Example: Poisson Mixtures
############################################################################
#  fu = mixing density evaluated at u
#   u = grid of (equi-spaced) points over which to mix
#   x = points at which the mixture is evaluated
#   p = parametric kernel (e.g., dnorm, dpois, etc)
# ... = some extra parameters for the kernel (e.g., a normal std dev)

mixture <- function(fu, u, x, p, ...) {
  # Ensure M is calculated correctly for each combination of x and u
  M <- outer(x, u, Vectorize(function(x, u) p(x, lambda=u, ...)))
  # Multiply each column of M by fu and sum rows, accounting for spacing in u
  out <- rowSums(M * rep(fu, each = length(x))) * (u[2] - u[1])
  return(out)
}

x <- 0:25
u <- seq(0, 50, len = 250)
fu <- dgamma(u, shape = 5, scale = 2)
gx <- mixture(fu = fu, x = x, u = u, p = dpois) # No need to pass lambda=u here since it's handled within mixture
plot(x, gx, type = "h", xlab = "x", ylab = "Poisson mixture")
lines(x + 0.2, dnbinom(x, size = 5, prob = 1/3), type = "h", col = 2)  # Comparing with a negative binomial
# Adding a legend
legend("topright", legend = c("Poisson Mixture", "Negative Binomial"), col = c(1, 2), lty = 1, 
       x.intersp = 0.5, y.intersp = 1.5, cex = 0.8, bty = "y")

############################################################################
## Example: Bayesian Analysis in a Binomial Model
############################################################################

n <- 40
theta <- 0.47
X <- rbinom(1, size = n, prob = theta)
p <- function(u) (8 / pi) * sqrt(0.25 - (u - 0.5)^2)
L <- function(u) dbinom(X, size = n, prob = u)
u <- seq(0, 1, len = 100)
post_vals <- p(u) * L(u)
K <- sum(post_vals * (u[2] - u[1]))
post <- post_vals / K

plot(u, post, type = "l", col = "blue", xlim = c(0, 1), 
     xlab = expression(theta), ylab = expression(pi[x](theta)), main = "Bayesian Example")
lines(u, p(u), col = "gray")
post_mean <- sum(u * post * (u[2] - u[1]))

abline(v = c(theta, post_mean, X / n), col = c("red", "green", "orange"))
legend("topright", legend = c("Posterior", "Prior", "Theta", "Post Mean", "X/n"), 
       col = c("blue", "gray", "red", "green", "orange"), lty = 1, cex = 0.8)

############################################################################
## Example: Maximum Marginal Likelihood (Example 5.1 in Givens & Hoeting)
############################################################################
y <- matrix(
  c(
    9, 6, 13, 9, 6, 11, 7, 8, 3, 4, 11, 1, 6, 0, 18, 15, 10, 6, 9, 4, 4, 10,
    12, 7, 18, 10, 7, 11, 10, 18, 3, 10, 10, 3, 7, 3, 18, 15, 14, 6, 9, 3, 13, 11,
    16, 10, 14, 12, 8, 12, 11, 19, 3, 11, 10, 2, 7, 3, 19, 15, 16, 7, 13, 4, 13, 13,
    17, 15, 21, 14, 9, 14, 12, 19, 7, 17, 15, 4, 9, 4, 22, 18, 17, 9, 16, 7, 16, 17,
    18, 16, 21, 15, 12, 16, 14, 22, 8, 18, 16, 5, 10, 6, 22, 19, 19, 10, 20, 9, 19, 21
  ), 
  nrow = 22, byrow = FALSE
)

# Reproduce results in Tables 5.2--5.4
y1 <- y[1,]
J <- length(y1)

# Vectorized function computation
f <- Vectorize(function(g) {
  b0 <- 1.804
  b1 <- 0.165
  sg <- 0.015
  lambda <- exp(b0 + b1 * (1:J) + g)
  sum((1:J) * (y1 - lambda)) * exp(sum(dpois(y1, lambda, log = TRUE))) * dnorm(g, mean = 0, sd = sg)
}, vectorize.args = "g")

G <- seq(-0.07, 0.085, len = 250)
fG <- f(G)
plot(G, fG, type = "l", xlab = expression(gamma[1]), ylab = "integrand")

integ_test <- function(m, ngrid) {
  sapply(ngrid, function(n) {
    G <- seq(-0.07, 0.085, len = n)
    fG <- f(G)
    est <- myint(fx = fG, x = G, m = m) * 1e05
    c(n_int = n, est = est)
  })
}

# Example call to integ_test
results <- integ_test(m = 2, ngrid = c(100, 250, 500))
results

############################################################################
# Actual optimization of the marginal log-likelihood in Equation (5.6)
############################################################################
log_like <- function(theta) {
  b0 <- theta[1]
  b1 <- theta[2]
  s <- theta[3]
  u <- seq(-0.1, 0.1, length.out = 100)
  o <- sapply(1:22, function(i) {
    v <- rowSums(sapply(1:5, function(j) dpois(y[i, j], exp(b0 + j * b1 + u), log = TRUE)))
    fu <- dnorm(u, mean = 0, sd = s) * exp(v)
    log(myint(fu, u))
  })
  -sum(o)
}

optim(par = c(1.804, 0.165, 0.015), fn = log_like, method = "BFGS")

############################################################################
#Example: Bayesian Analysis of Binomial
############################################################################
# semicircle distribution function
semicircle_distr <- function(theta) {
  (8/pi) * sqrt((1/4) - (theta - 1/2)^2)
}

# a sequence of theta values
theta <- seq(0, 1, length.out = 500)
# the density values
density <- semicircle_distr(theta)
# data frame for ggplot
data <- data.frame(theta, density)

# Plot of the distribution
ggplot(data, aes(x = theta, y = density)) +
  geom_line(color = "blue") +
  geom_area(fill = "skyblue", alpha = 0.4) +
  labs(title = "Prior Distribution for Theta",
       x = expression(theta),
       y = "density") +
  theme_minimal()

# necessary library
library(stats4)

# prior distribution function for theta
prior <- function(theta) {
  ifelse(theta >= 0 & theta <= 1, (8/pi) * sqrt((1/4) - (theta - 1/2)^2), 0)
}

# likelihood function for the binomial distribution
likeli <- function(theta, x, n) {
  theta^x * (1 - theta)^(n - x)
}

# posterior density function
post <- function(theta, x, n) {
  likeli(theta, x, n) * prior(theta)
}

# Numerical integration of the posterior density to normalize it
norm_post <- function(x, n) {
  integrate(function(theta) post(theta, x, n), lower = 0, upper = 1)$value
}

# the posterior mean (Bayes estimate) for theta
bayes_est <- function(x, n) {
  integ <- integrate(function(theta) theta * post(theta, x, n), lower = 0, upper = 1)$value
  integ / norm_post(x, n)
}

# Given values
n <- 20
x <- 12

# Bayes estimate of theta
bayes_est_theta <- bayes_est(x, n)
bayes_est_theta



